package edu.mayo.bior.catalog;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.util.List;

import static edu.mayo.bior.util.TestUtil.buildEmptyCatalog;
import static edu.mayo.bior.util.TestUtil.mkdir;
import static junit.framework.Assert.assertEquals;
import static junit.framework.Assert.assertNull;
import static junit.framework.Assert.fail;
import static org.junit.Assert.assertTrue;


/**
 * Test CatalogFileUtils
 */
public class CatalogFileUtilsTest
{
   @Rule
   public TemporaryFolder temporaryFolder = new TemporaryFolder();

   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   @Test
   public void testBufferedReaderOnBgz() throws IOException
   {
      String path = "src/test/resources/testData/verification/catalog_with_2_fields/catalog.tsv.bgz";
      BufferedReader reader = CatalogFileUtils.getBufferedReader(path);
      assertEquals(1, countLines(reader));
   }

   @Test
   public void testBufferedReaderonText() throws IOException
   {
      String path = "src/test/resources/testData/verification/mini_catalog.json";
      BufferedReader reader = CatalogFileUtils.getBufferedReader(path);
      assertEquals(2, countLines(reader));
   }

   @Test
   public void testFileNotExist()
   {
      assertNull(CatalogFileUtils.getBufferedReader("filethatcouldntpossiblyexist"));
   }

   @Test
   public void testFileIsDir() throws IOException
   {
      assertNull(CatalogFileUtils.getBufferedReader(temporaryFolder.newFolder().getPath()));
   }

   @Test
   public void testNotReadableFile() throws IOException
   {
      File file = temporaryFolder.newFile();
      if (!file.setReadable(false))
      {
         String msg = String.format("Couldn't set temp file '%s' to unreadable to test", file.getPath());
         fail(msg);
      }
      assertNull(CatalogFileUtils.getBufferedReader(file.getPath()));
   }

   @Test
   public void testZeroLengthFile() throws IOException
   {
      BufferedReader reader = CatalogFileUtils.getBufferedReader(temporaryFolder.newFile().getPath());
      assertEquals(0, countLines(reader));
   }

   private int countLines(BufferedReader reader) throws IOException
   {
      int count = 0;
      while (reader.readLine() != null)
      {
         count++;
      }
      return count;
   }

   @Test
   public void findAllTsvBgzFilesExhaustive() throws Exception {
      File rootDir = temporaryFolder.newFolder("catalog_root_dir");

      File fooV1CatalogFile = buildEmptyCatalog(mkdir(rootDir,"FOO_V1_GRCh37"), "FOO", "v1", "GRCh37", "v1 desc", "v1 dataset", "", temporaryFolder);
      File fooV2CatalogFile = buildEmptyCatalog(mkdir(rootDir,"FOO_V2_GRCh37"), "FOO", "v2", "GRCh37", "v1 desc", "v1 dataset", "", temporaryFolder);

      // nested catalog folder
      File nestedCatalogFile = buildEmptyCatalog(mkdir(fooV1CatalogFile.getParentFile(),"nested"), "nested", "v1", "GRCh37", "v1 desc", "v1 dataset", "", temporaryFolder);

      List<File> catalogFiles = CatalogFileUtils.findAllTsvBgzFiles(rootDir, true);
      assertEquals(3, catalogFiles.size());
      assertTrue(catalogFiles.contains(fooV1CatalogFile));
      assertTrue(catalogFiles.contains(fooV2CatalogFile));
      assertTrue(catalogFiles.contains(nestedCatalogFile));
   }

   @Test
   public void findAllTsvBgzFilesNonExhaustive() throws Exception {
      File rootDir = temporaryFolder.newFolder("catalog_root_dir");

      File fooV1CatalogFile = buildEmptyCatalog(mkdir(rootDir,"FOO_V1_GRCh37"), "FOO", "v1", "GRCh37", "v1 desc", "v1 dataset", "", temporaryFolder);
      File fooV2CatalogFile = buildEmptyCatalog(mkdir(rootDir,"FOO_V2_GRCh37"), "FOO", "v2", "GRCh37", "v1 desc", "v1 dataset", "", temporaryFolder);

      // nested catalog folder
      buildEmptyCatalog(mkdir(fooV1CatalogFile.getParentFile(),"nested"), "nested", "v1", "GRCh37", "v1 desc", "v1 dataset", "", temporaryFolder);

      List<File> catalogFiles = CatalogFileUtils.findAllTsvBgzFiles(rootDir, false);
      assertEquals(2, catalogFiles.size());
      assertTrue(catalogFiles.contains(fooV1CatalogFile));
      assertTrue(catalogFiles.contains(fooV2CatalogFile));
   }

   @Test
   public void findAllTsvBgzFilesNoCatalogs() throws Exception {
      File rootDir = temporaryFolder.newFolder("catalog_root_dir");

      List<File> catalogFiles = CatalogFileUtils.findAllTsvBgzFiles(rootDir, false);
      assertEquals(0, catalogFiles.size());
   }
}
