package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.catalog.DataSourceKey;


/** Test the merging of just the datasource.properties file
 *  @author  Michael Meiners (m054457) - 2016-06-01 */
public class MergeDataSourcePropertiesTest {

	
	private File mTargetDir;
	private String mCatalogShortName = "x";
	//private File mPreviousCatalogDataSource = null;  // None this first time
	private StepLogger mStepLogger;
	TemporaryFolder mTempFolder;
	private BuildInfo mBuildInfo;
	
	
	@Before
	public void beforeEach() throws IOException, BuildCatalogStepInputException {
		mTempFolder = new TemporaryFolder();
		mTempFolder.create();
		mTargetDir = mTempFolder.newFolder();
		mStepLogger = new StepLogger(BuildStepKey.MERGE_PROP_FILES.toString(), mTargetDir);
		copyPropFilesToTemp();
		
		Map<String, String> buildInfoMap = new HashMap<String, String>();
		buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), mTargetDir.getCanonicalPath());
		buildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(),   mCatalogShortName);
		//buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), null); //mPreviousCatalogDataSource.getCanonicalPath());
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(), "dbSNP");
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), "GRCh37.p14");
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), "143");
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_RELEASE_DATE.name(), "2018-07-13");
		// These shouldn't be used
		File makeJsonScript = new File("src/test/resources/buildCatalog/makeJsonStep.sh");
		buildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), makeJsonScript.getCanonicalPath());
		buildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--noflags");
		mBuildInfo = new BuildInfo(buildInfoMap);
	}
	
	private void copyPropFilesToTemp() throws IOException {
		FileUtils.copyDirectory(new File("src/test/resources/buildCatalog/mergeProps"), mTargetDir);
	}
	
	private File copyPreviousCatalog() throws IOException {
		File prevCtgDir = mTempFolder.newFolder();
		File previousCatalogDataSource = new File(prevCtgDir, "x.datasource.properties.previousCatalog");
		FileUtils.copyFile(new File("src/test/resources/buildCatalog/mergeProps_previousCatalog/x.datasource.properties"), previousCatalogDataSource);
		mBuildInfo.getMap().put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), previousCatalogDataSource.getCanonicalPath());
		return previousCatalogDataSource;
	}
	
	@Test
	public void testNewCatalog() throws Exception {
		MergeDataSourceProperties merge = new MergeDataSourceProperties(mBuildInfo, mStepLogger);
		merge.mergeDatasourceProperties();
		
		// Verify that all files are present
		MergeColumnsTsvTest.assertDirContainsOnly(mTargetDir, Arrays.asList(
				BuildCatalog.BUILD_SUBDIR,
				"x.columns.tsv",
				"x.columns.tsv.blacklist",
				"x.columns.tsv.blacklist.biorweb",
				"x.datasource.properties" ));
		MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR), Arrays.asList(
				"progress",
				"x.columns.tsv.default",
				// Backup occurs because file is change due to build_info.txt data
				"x.datasource.properties.01",
				"x.datasource.properties.default"  ));
		File mergeLog     = new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildStepKey.MERGE_PROP_FILES.toString() + ".log");
		File mergeSummary = new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildStepKey.MERGE_PROP_FILES.toString() + ".summary");
		MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress"), Arrays.asList(mergeLog.getName(), mergeSummary.getName()));
		
		// Now verify contents of the datasource (generic, empty for most fields) files and logs
		String expectedDataSourceProps = 
			"### Datasource properties file for Catalog - x.  Please fill in the descriptions to the keys below.\n" +
			"## Short name that should be unique (or mostly unique except for fixes to existing catalog) across all catalogs. Ex: dbSNP_142_GRCh37p13\n" +
			// Filled in by BuildInfo object
			"ShortUniqueName=dbSNP_143_GRCh37p14\n" +
			"## Description of catalog.  Ex: NCBI's dbSNP Variant Database\n" +
			// No previous catalog was supplied, so blank
			"Description=\n" +
			"## Source of data, without point release, etc.  Ex: dbSNP\n" +
			// Filled in by BuildInfo object
			"Source=dbSNP\n" +
			"## Type of data.  Ex: Variants\n" +
			// No previous catalog was supplied, so blank
			"Dataset=\n" +
			"## Version of the data source.  Ex: 142\n" +
			// Filled in by BuildInfo object
			"Version=143\n" +
			"## The Genome build/assembly.  Ex: GRCh37.p13\n" +
			// Filled in by BuildInfo object
			"Build=GRCh37.p14\n" +
			"## The BioR catalog compatibility format.  Ex: 1.1.1\n" +
			"Format=1.1.1\n" +
			"## The release date of the data source from the provider (not the BioR build date).  Ex: 2018-07-10\n" + 
			// Filled in by BuildInfo object
			"DataSourceReleaseDate=2018-07-13\n";
		File datasrcFile = new File(mTargetDir, mCatalogShortName + ".datasource.properties");
		String expectedSummary = // Nearly same as log, but without the NOTEs
			"Some changes made to '" + mTargetDir.getCanonicalPath() + "/x.datasource.properties'\n" +
			"Datasource.properties file '" + mTargetDir.getCanonicalPath() + "/x.datasource.properties' has some issues. See verify output for details.\n";

		String expectedLog =
				"Note: (datasource.properties): Value for key 'ShortUniqueName' coming from build_info.txt file with value: dbSNP_143_GRCh37p14\n" +
				// Description not found because no previous catalog supplied
				"Warning: (datasource.properties): Key 'Description' missing, or value is empty.  Attempting to add key and value...\n" +
				"    Could not find a value in the previous catalog or defaults, so setting it to empty\n" +
				"Note: (datasource.properties): Value for key 'Source' coming from build_info.txt file with value: dbSNP\n" +
				// Dataset not found because no previous catalog supplied
				"Warning: (datasource.properties): Key 'Dataset' missing, or value is empty.  Attempting to add key and value...\n" +
				"    Could not find a value in the previous catalog or defaults, so setting it to empty\n" +
				"Note: (datasource.properties): Value for key 'Version' coming from build_info.txt file with value: 143\n" +
				"Note: (datasource.properties): Value for key 'Build' coming from build_info.txt file with value: GRCh37.p14\n" +
				"Note: (datasource.properties): Value for key 'DataSourceReleaseDate' coming from build_info.txt file with value: 2018-07-13\n" +
				expectedSummary;

		
		String actualDataSourceProps = FileUtils.readFileToString(new File(mTargetDir, "x.datasource.properties"));
		String actualLog = MergeColumnsTsvTest.removeTimestamp(FileUtils.readFileToString(mergeLog));
		String actualSummary = FileUtils.readFileToString(mergeSummary);
		
		assertEquals(expectedDataSourceProps, actualDataSourceProps);
		assertEquals(expectedLog, actualLog);
		assertEquals(expectedSummary, actualSummary);
	}


	@Test
	public void testMergeWithPreviousCatalog() throws Exception {
		copyPreviousCatalog();
		
		MergeDataSourceProperties merge = new MergeDataSourceProperties(mBuildInfo, mStepLogger);
		merge.mergeDatasourceProperties();
		
		// VERIFY_CATALOG that all files are present
		MergeColumnsTsvTest.assertDirContainsOnly(mTargetDir, Arrays.asList(
				BuildCatalog.BUILD_SUBDIR,
				"x.columns.tsv",
				"x.columns.tsv.blacklist",
				"x.columns.tsv.blacklist.biorweb",
				"x.datasource.properties"
				));
		MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR), Arrays.asList(
					"progress",
					"x.columns.tsv.default",
					"x.datasource.properties.01",
					"x.datasource.properties.default"
				));
		File mergeLog     = new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildStepKey.MERGE_PROP_FILES.toString() + ".log");
		File mergeSummary = new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildStepKey.MERGE_PROP_FILES.toString() + ".summary");
		MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress"), Arrays.asList(
						mergeLog.getName(),
						mergeSummary.getName()
						));
		
		// Now verify contents of the datasource (generic, empty for most fields) files and logs
		String expectedDataSourceProps = 
			"### Datasource properties file for Catalog - x.  Please fill in the descriptions to the keys below.\n" +
			"## Short name that should be unique (or mostly unique except for fixes to existing catalog) across all catalogs. Ex: dbSNP_142_GRCh37p13\n" +
			// Coming from BuildInfo object instead of previous catalog
			"ShortUniqueName=dbSNP_143_GRCh37p14\n" +
			"## Description of catalog.  Ex: NCBI's dbSNP Variant Database\n" +
			// From previous catalog
			"Description=NCBI's dbSNP Variant Database\n" +
			"## Source of data, without point release, etc.  Ex: dbSNP\n" +
			"Source=dbSNP\n" +
			"## Type of data.  Ex: Variants\n" +
			// From previous catalog
			"Dataset=Variants\n" +
			"## Version of the data source.  Ex: 142\n" +
			// Coming from BuildInfo object instead of previous catalog
			"Version=143\n" +
			"## The Genome build/assembly.  Ex: GRCh37.p13\n" +
			// Coming from BuildInfo object instead of previous catalog
			"Build=GRCh37.p14\n" +
			"## The BioR catalog compatibility format.  Ex: 1.1.1\n" +
			"Format=1.1.1\n" +
			"## The release date of the data source from the provider (not the BioR build date).  Ex: 2018-07-10\n" +
			// Coming from BuildInfo object instead of previous catalog
			"DataSourceReleaseDate=2018-07-13\n";
		File datasrcFile = new File(mTargetDir, mCatalogShortName + ".datasource.properties");
		String expectedSummary = // Same as log except without the Notes
			"Some changes made to '" + mTargetDir.getCanonicalPath() + "/x.datasource.properties'\n";
		String expectedLog =
				"Note: (datasource.properties): Value for key 'ShortUniqueName' coming from build_info.txt file with value: dbSNP_143_GRCh37p14\n" +
				"Warning: (datasource.properties): Key 'Description' missing, or value is empty.  Attempting to add key and value...\n" +
				"    Using the previous catalog key and value: NCBI's dbSNP Variant Database\n" +
				"Note: (datasource.properties): Value for key 'Source' coming from build_info.txt file with value: dbSNP\n" +
				"Warning: (datasource.properties): Key 'Dataset' missing, or value is empty.  Attempting to add key and value...\n" +
				"    Using the previous catalog key and value: Variants\n" +
				"Note: (datasource.properties): Value for key 'Version' coming from build_info.txt file with value: 143\n" +
				"Note: (datasource.properties): Value for key 'Build' coming from build_info.txt file with value: GRCh37.p14\n" +
				"Note: (datasource.properties): Value for key 'DataSourceReleaseDate' coming from build_info.txt file with value: 2018-07-13\n" +
				"Note: (datasource.properties): 'ShortUniqueName' value is different between current ('dbSNP_143_GRCh37p14') and previous ('dbSNP_142_GRCh37p13') catalog.\n" +
				"Note: (datasource.properties): 'Version' value is different between current ('143') and previous ('142') catalog.\n" +
				"Note: (datasource.properties): 'Build' value is different between current ('GRCh37.p14') and previous ('GRCh37.p13') catalog.\n" +
				expectedSummary;

		String actualDataSourceProps = FileUtils.readFileToString(new File(mTargetDir, "x.datasource.properties"));
		String actualLog = MergeColumnsTsvTest.removeTimestamp(FileUtils.readFileToString(mergeLog));
		String actualSummary = FileUtils.readFileToString(mergeSummary);
		
		assertEquals(expectedDataSourceProps, actualDataSourceProps);
		assertEquals(expectedLog, actualLog);
		assertEquals(expectedSummary, actualSummary);
	}
	
	// ================== Test the DATA_SOURCE_RELEASE_DATE ==============================================

	@Test
	/** The DATA_SOURCE_RELEASE_DATE should be picked up only from the new catalog's build_info.txt file, and NOT from the previous catalog */
	public void testWithPreviousCatalog_dataSourceReleaseDate() throws Exception {
		// If value from previous catalog is null, then it is missing.  If "", then the key is there, but the value is ""
		String[] previousCtgReleaseDate = { null,  null,		  "",  "",            "2015-05-15",  "2015-05-15" };
		String[] buildinfoReleaseDate	= { "",    "2017-07-07",  "",  "2017-07-07",  "",            "2017-07-07" };
		String[] expectedReleaseDate  	= { "",    "2017-07-07",  "",  "2017-07-07",  "",            "2017-07-07" };
		
		for(int i=0; i < previousCtgReleaseDate.length; i++) {
			beforeEach(); // Reset so we get fresh files each time
			File prevCtgDatasourcePropsFile = copyPreviousCatalog();
			
			// Set value in both the previous catalog's datasource.properties, and in the new BuildInfo object
			setDataSourcePropsVal(prevCtgDatasourcePropsFile, DataSourceKey.DataSourceReleaseDate, previousCtgReleaseDate[i]);
			mBuildInfo.getMap().put(BuildInfoKey.DATA_SOURCE_RELEASE_DATE.name(), buildinfoReleaseDate[i]);
			
			MergeDataSourceProperties merge = new MergeDataSourceProperties(mBuildInfo, mStepLogger);
			merge.mergeDatasourceProperties();
			
			String actual = getDataSourceVal(new File(mTargetDir, "x.datasource.properties"), DataSourceKey.DataSourceReleaseDate);
			String msg = BuildInfoKey.DATA_SOURCE_RELEASE_DATE.name() + " (idx: " + i + "):\n"
					+  "Prev:                     [" + previousCtgReleaseDate[i] + "]\n"
					+  "New (in build_info.txt):  [" + buildinfoReleaseDate[i] + "]\n"
					+  "Actual:                   [" + actual + "]";
			assertEquals(msg, expectedReleaseDate[i], actual);
		}
	}

	@Test
	/** The Dataset should be picked up from previous catalog if provided, but preference given to build_info.txt value */
	public void testWithPreviousCatalog_dataset() throws Exception {
		// If value from previous catalog is null, then it is missing.  If "", then the key is there, but the value is ""
		String[] previousCtgDataset = { null,  null,	  "",  "",       "Variants",  "Variants" };
		String[] buildinfoDataset   = { "",    "Genes",   "",  "Genes",  "",          "Genes" };
		String[] expectedDataset  	= { "",    "Genes",   "",  "Genes",  "Variants",  "Genes" };
		
		for(int i=0; i < previousCtgDataset.length; i++) {
			beforeEach(); // Reset so we get fresh files each time
			File prevCtgDatasourcePropsFile = copyPreviousCatalog();
			
			// Set value in both the previous catalog's datasource.properties, and in the new BuildInfo object
			setDataSourcePropsVal(prevCtgDatasourcePropsFile, DataSourceKey.Dataset, previousCtgDataset[i]);
			mBuildInfo.getMap().put(BuildInfoKey.DATA_SOURCE_DATASET.name(), buildinfoDataset[i]);
			
			MergeDataSourceProperties merge = new MergeDataSourceProperties(mBuildInfo, mStepLogger);
			merge.mergeDatasourceProperties();
			
			String actual = getDataSourceVal(new File(mTargetDir, "x.datasource.properties"), DataSourceKey.Dataset);
			String msg = BuildInfoKey.DATA_SOURCE_DATASET.name() + " (idx: " + i + "):\n"
					+  "Prev:                     [" + previousCtgDataset[i] + "]\n"
					+  "New (in build_info.txt):  [" + buildinfoDataset[i] + "]\n"
					+  "Actual:                   [" + actual + "]";
			assertEquals(msg, expectedDataset[i], actual);
		}
	}
	
	@Test
	public void testShortUniqueName() throws BuildCatalogStepInputException, IOException {
		// Create a make_json script and make it executable
		File makeJsonScript = new File(mTargetDir, "make_json.sh");
		makeJsonScript.createNewFile();
		makeJsonScript.setExecutable(true);
		
		Map<String,String> keyValMap = new HashMap<String,String>();
		keyValMap.put(BuildInfoKey.DATA_SOURCE.toString(), 			"omim");
		keyValMap.put(BuildInfoKey.DATA_SOURCE_VERSION.toString(), 	"2018_09_05");
		keyValMap.put(BuildInfoKey.DATA_SOURCE_BUILD.toString(), 	"GRCh37.p13");
		keyValMap.put(BuildInfoKey.MAKE_JSON_ARGS.toString(), 		"");
		keyValMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.toString(),makeJsonScript.getCanonicalPath());
		keyValMap.put(BuildInfoKey.CATALOG_PREFIX.toString(), 		"omim_genes");
		
		BuildInfo buildInfo = new BuildInfo(keyValMap);
		MergeDataSourceProperties dataSrcProps = new MergeDataSourceProperties(null, null);
		String actual = dataSrcProps.getShortUniqueName(buildInfo);
		final String EXPECTED = "omim_20180905_GRCh37p13";
		
		assertEquals(EXPECTED, actual);
	}
	
	private String getDataSourceVal(File prevCtgDatasourcePropsFile, DataSourceKey key) throws IOException {
		Properties props = new Properties();
		FileInputStream fin = null;
		String val = null;
		try {
			fin = new FileInputStream(prevCtgDatasourcePropsFile);
			props.load(fin);
			val = props.getProperty(key.name());
		} finally {
			if( fin != null ) 
				fin.close();
		}
		return val;
	}

	/** Set the DATA_SOURCE_RELEASE_DATE in the datasource.properties or build_info.txt file.  If null, then remove it.  If "", then set to blank.  Else fill it in 
	 * @throws IOException */ 
	private void setDataSourcePropsVal(File dataSourcePropsOrBuildInfo, DataSourceKey key, String val) throws IOException {
		String contents = FileUtils.readFileToString(dataSourcePropsOrBuildInfo);
		// Remove existing (NOTE: "(?m)" will turn on multi-line mode to match a word at start of every line.  This affects only "^" and "$"
		// See: https://howtodoinjava.com/regex/java-regex-match-something-at-the-start-andor-the-end-of-line/
		// section: "Regex to Match word at the start of line"
		String regex = "(?m)^" + key.name() + "=.*$";
		contents = contents.replaceAll(regex, "");
		if( val != null ) {
			contents = contents + "\n" + key.name() + "=" + val;
		}
		FileUtils.write(dataSourcePropsOrBuildInfo, contents);
	}
}
