package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.pipeline.exception.CatalogMetadataInputException;

public class MergeColumnsTsvTest {

	private File mTargetDir;
	private File mTempDir;
	private String mCatalogShortName = "x";
	private File mPreviousCatalogColumnsTsv = null;  // None this first time
	private StepLogger mStepLogger;

	
	@Before
	public void beforeEach() throws IOException {
		TemporaryFolder tempFolder = new TemporaryFolder();
		tempFolder.create();
		mTargetDir = tempFolder.newFolder();
		mTempDir = tempFolder.newFolder();
		mStepLogger = new StepLogger(BuildStepKey.MERGE_PROP_FILES.toString(), mTargetDir);
		copyPropFilesToTemp();
	}
	
	private void copyPropFilesToTemp() throws IOException {
		FileUtils.copyDirectory(new File("src/test/resources/buildCatalog/mergeProps"), mTargetDir);
	}

	
	@Test
	public void newCatalogTest() throws IOException, CatalogMetadataInputException {
		MergeColumnsTsv merge = new MergeColumnsTsv(mTargetDir, mCatalogShortName, mPreviousCatalogColumnsTsv, mStepLogger);
		merge.mergeColumnsTsv();
		
		// Verify that the columns.tsv is unchanged from the input
		File expectedColumnsTsv = new File("src/test/resources/buildCatalog/mergeProps/x.columns.tsv");
		File actualColumnsTsv = new File(mTargetDir, "x.columns.tsv");
		assertEquals(FileUtils.readFileToString(expectedColumnsTsv), FileUtils.readFileToString(actualColumnsTsv));
		
		// Verify contents of target directory ONLY contains:
		assertDirContainsOnly(mTargetDir, Arrays.asList(
				BuildCatalog.BUILD_SUBDIR,
				"x.columns.tsv",
				"x.columns.tsv.blacklist",
				"x.columns.tsv.blacklist.biorweb",
				"x.datasource.properties"));
		
		// Verify contents of log and summary
			String expectedLog =
				"No changes made to '" + mTargetDir + "/x.columns.tsv'\n" +
				"Columns.tsv file '" + mTargetDir + "/x.columns.tsv' has some issues. See verify output for details.\n";
		String expectedSummary = expectedLog;
		String logFilePrefix = BuildStepKey.MERGE_PROP_FILES.toString();
		String actualLog = removeTimestamp(FileUtils.readFileToString(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + logFilePrefix + ".log")));
		String actualSummary = FileUtils.readFileToString(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + logFilePrefix + ".summary"));
		assertEquals(expectedLog, actualLog);
		assertEquals(expectedSummary, actualSummary);
	}
	
	@Test
	/** Test the case where the user has modified the columns.tsv from the default by changing Description and HumanReadableName.
	 *  First copy the original properties files over to the temp dir,
	 *  Then copy over the modified x.columns.tsv that contains updated descriptions and HumanReadableNames.
	 *  Copy the x.columns.tsv file and use as previous catalog, but remove a line from it.
	 *  RESULTS:
	 *    - We should see a log statement and summary line about the missing line in the previous catalog x.columns.tsv file
	 *    - No backup columns tsv's since output is same
	 */
	public void manualChgToHumanReadableNames() throws IOException, CatalogMetadataInputException {
		// Copy the file with the manually added descriptions and human readable names over the existing columns.tsv in the temp dir
		FileUtils.copyFile(new File("src/test/resources/buildCatalog/mergeProps_previousCatalog/x.columns.tsv"), new File(mTargetDir, "x.columns.tsv"));
		MergeColumnsTsv merge = new MergeColumnsTsv(mTargetDir, mCatalogShortName, mPreviousCatalogColumnsTsv, mStepLogger);
		merge.mergeColumnsTsv();
		
		// Verify that the columns.tsv is unchanged from the input
		File expectedColumnsTsv = new File("src/test/resources/buildCatalog/mergeProps_previousCatalog/x.columns.tsv");
		File actualColumnsTsv = new File(mTargetDir, "x.columns.tsv");
		assertEquals(FileUtils.readFileToString(expectedColumnsTsv), FileUtils.readFileToString(actualColumnsTsv));
		
		// Verify contents of target directory ONLY contains:
		assertDirContainsOnly(mTargetDir, Arrays.asList(
				BuildCatalog.BUILD_SUBDIR,
				"x.columns.tsv",
				"x.columns.tsv.blacklist",
				"x.columns.tsv.blacklist.biorweb",
				"x.datasource.properties"));

		// Verify contents of log and summary
		String logFilePrefix = BuildStepKey.MERGE_PROP_FILES.toString();
		// Verify contents of log and summary
		String expectedLog =
			"No changes made to '" + mTargetDir + "/x.columns.tsv'\n";
		String expectedSummary = expectedLog;
		String actualLog = removeTimestamp(FileUtils.readFileToString(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + logFilePrefix + ".log")));
		String actualSummary = FileUtils.readFileToString(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + logFilePrefix + ".summary"));
		assertEquals(expectedLog, actualLog);
		assertEquals(expectedSummary, actualSummary);
	}

	
	@Test
	/** Test the case where the previous catalog has the Description and HumanReadableName that we want, but the current catalog has not changed yet.
	 *  Also, add a column to the previous catalog, and remove one line from the defaults.
	 *  
	 *  First copy the original properties files over to the temp dir,
	 *  Point to modified x.columns.tsv as the previous catalog that contains updated descriptions and HumanReadableNames.
	 *    - add a line to it
	 * 
	 *  Then remove one line from the x.columns.tsv.default file.
	 *  
	 *  RESULTS:
	 *    - We should see a log and summary statement about the missing line in the x.columns.tsv.default file  (or extra line in x.columns.tsv)
	 *    - We Should see a log and summary statement about extra line in previous catalog's columns.tsv
	 *    - We Should have a backup since the Description and HumanReadableName were modified (to the values from the previous catalog)
	 */
	public void previousCatalogHasDescriptionAndHumanReadableNameWeWant() throws IOException, CatalogMetadataInputException {
		mPreviousCatalogColumnsTsv = new File(mTempDir, "previousCatalog.columns.tsv");
		// Copy the file with the manually added descriptions and human readable names to the "previous catalog" columns.tsv
		FileUtils.copyFile(new File("src/test/resources/buildCatalog/mergeProps_previousCatalog/x.columns.tsv"), mPreviousCatalogColumnsTsv);
		
		// Add a new line to the previous columns.tsv:
		String columnsTsv = FileUtils.readFileToString(mPreviousCatalogColumnsTsv);
		columnsTsv = columnsTsv + "extraKey	String	1	An extra key that exists in the previous catalog columns.tsv	Extra Key\n";
		FileUtils.write(mPreviousCatalogColumnsTsv, columnsTsv);
		
		// Remove a line from the default columns.tsv  (for the "CHR" key)
		File defaultFile = new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/x.columns.tsv.default");
		String defaultColumnsTsv = FileUtils.readFileToString(defaultFile);
		FileUtils.write(defaultFile, defaultColumnsTsv.replace("CHR	String	1		CHR\n", ""));

		MergeColumnsTsv merge = new MergeColumnsTsv(mTargetDir, mCatalogShortName, mPreviousCatalogColumnsTsv, mStepLogger);
		merge.mergeColumnsTsv();
		
		// Verify that the columns.tsv now has description and HumanReadableName from the previous catalog
		File expectedColumnsTsv = new File("src/test/resources/buildCatalog/mergeProps_previousCatalog/x.columns.tsv");
		File actualColumnsTsv = new File(mTargetDir, "x.columns.tsv");
		assertEquals(FileUtils.readFileToString(expectedColumnsTsv), FileUtils.readFileToString(actualColumnsTsv));
		
		// Verify contents of target directory ONLY contains:
		assertDirContainsOnly(mTargetDir, Arrays.asList(
				BuildCatalog.BUILD_SUBDIR,
				"x.columns.tsv",
				"x.columns.tsv.blacklist",
				"x.columns.tsv.blacklist.biorweb",
				"x.datasource.properties"
				));
		
		// Verify contents of target/[BuildCatalog.BUILD_SUBDIR] directory ONLY contains:
		assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR), Arrays.asList(
				"progress",
				"x.columns.tsv.01",
				"x.columns.tsv.default",
				"x.datasource.properties.default"
				));
		

		// Verify contents of log and summary
		String logFilePrefix = BuildStepKey.MERGE_PROP_FILES.toString();
		// Verify contents of log and summary
		String expectedLog = 
				"Warning: (columns.tsv): Using the previous catalog's description for column 'ALTS'\n" +
				"Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column 'ALTS'\n" +
				"Warning: (columns.tsv): Using the previous catalog's description for column 'CHR'\n" +
				"Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column 'CHR'\n" +
				"Warning: (columns.tsv): Extra key?  'CHR' is in the current columns.tsv, but was not detected in the catalog (not in columns.tsv.defaults).\n" +
				"Warning: (columns.tsv): Extra key?  'CHR' is in the current columns.tsv, but not in the previous catalog's columns.tsv.\n" +
				"Warning: (columns.tsv): Using the previous catalog's description for column 'GENENAME'\n" +
				"Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column 'GENENAME'\n" +
				// POS and REF already have HumanReadableNames assigned, so no warnings from those.  Continue with RSID
				"Warning: (columns.tsv): Using the previous catalog's description for column 'RSID'\n" +
				"Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column 'RSID'\n" +
				"Warning: (columns.tsv): These columns were present in the previous catalog's columns.tsv but not in the current columns.tsv:\n" +
				"    extraKey	String	1	An extra key that exists in the previous catalog columns.tsv	Extra Key\n" +
				"Some changes made to '" + mTargetDir.getPath() + "/x.columns.tsv'\n";
		String expectedSummary =
				"Some changes made to '" + mTargetDir.getPath() + "/x.columns.tsv'\n";
		String actualLog = removeTimestamp(FileUtils.readFileToString(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + logFilePrefix + ".log")));
		String actualSummary = FileUtils.readFileToString(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + logFilePrefix + ".summary"));
		assertEquals(expectedLog, actualLog);
		assertEquals(expectedSummary, actualSummary);
	}

	
	protected static String removeTimestamp(String log) {
		return StepLoggerTest.removeFirstXCharsFromEachLine(log, StepLogger.DATE_FORMAT_STR.length());
	}
	
	protected static void assertDirContainsOnly(File targetDir, List<String> expectedFileList) throws IOException {
		if( targetDir == null )
			throw new IllegalArgumentException("The target directory is null");
		else if( ! targetDir.exists() )
			throw new IllegalArgumentException("The target directory did not exist: " + targetDir.getCanonicalPath());
		
		Collections.sort(expectedFileList);
		File[] files = removeSvnDirs(targetDir.listFiles());
		Arrays.sort(files);
		String dirCompare = dirCompare(expectedFileList, filesToStrList(files));
		assertEquals(dirCompare, expectedFileList.size(), files.length);
		for(int i=0; i < expectedFileList.size(); i++) {
			assertEquals(dirCompare, expectedFileList.get(i), files[i].getName());
		}
	}
	
	/** Jenkins will have a ".svn" directory in each directory, which will mess up our count.  So, let's remove it from the list */
	private static File[] removeSvnDirs(File[] files) {
		List<File> fileListWithoutSvn = new ArrayList<File>();
		for(File file : files) {
			if( ! file.getName().equals(".svn") )
				fileListWithoutSvn.add(file);
		}
		return fileListWithoutSvn.toArray(new File[fileListWithoutSvn.size()]);
	}

	private static List<String> filesToStrList(File[] files) {
		List<String> list = new ArrayList<String>();
		for(File f : files) {
			list.add(f.getName());
		}
		return list;
	}
	
	private static String dirCompare(List<String> expectedFiles, List<String> actualFiles) {
		StringBuilder str = new StringBuilder("\nExpected: --------------------------\n");
		for(String expectedFile : expectedFiles) {
			str.append("    " + expectedFile + "\n");
		}
		str.append("\nActual: --------------------------\n");
		for(String actualFile : actualFiles) {
			str.append("    " + actualFile + "\n");
		}
		str.append("------------------------------------\n");
		return str.toString();
	}
}
