package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.*;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.catalog.HumanBuildAssembly;

public class CreateCatalogStepTest
{

   private String mTempDirName = null;
   private String reverseChrSortOrderFile = "src/test/resources/createCatalog/reverse_chr_sort_order.txt";

   private String mTargetDirName = null;
   private Map<String, String> mBuildInfoMap = new HashMap<String, String>();

   @Before
   public void before() throws IOException
   {

      TemporaryFolder outFolder = new TemporaryFolder();
      outFolder.create();
      mTargetDirName = outFolder.getRoot().getAbsolutePath();

      TemporaryFolder tmpFolder = new TemporaryFolder();
      tmpFolder.create();
      mTempDirName = tmpFolder.getRoot().getAbsolutePath();

      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), "src/test/resources/createCatalog/tjson_input.tsv");
      mBuildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), mTargetDirName);
      mBuildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), mTempDirName);
      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), HumanBuildAssembly.GRCh37.name());
      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), "2");
      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(), "x");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), new File("src/test/resources/buildCatalog/makeJsonStep.sh").getCanonicalPath());

   }

   @Test
   public void testDefaults() throws IOException, BuildCatalogStepInputException
   {
      String catalogPrefix = "createStepTest_sortedDefaultCatalog";
      ensureCatalogDeleted(catalogPrefix);

      mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), catalogPrefix);
      BuildInfo buildInfo = new BuildInfo(mBuildInfoMap);

      try
      {
         CreateStep createCatalogStep = new CreateStep(buildInfo, new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), new File(mTargetDirName)));
         createCatalogStep.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         //System.err.println("input exception occurred: " + inputE);
         fail("Unexpected exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         //System.err.println("exec exception occurred: " + execE);
         fail("Unexpected exception occurred: " + execE);
      }

      String catalogOutFilePath = mTargetDirName + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
      String catalogTabixIndexPath = catalogOutFilePath + ".tbi";

      File catOutFile = new File(catalogOutFilePath);
      File tabixIdxFile = new File(catalogTabixIndexPath);

      assertTrue(catOutFile.exists() && catOutFile.length() > 0);
      assertTrue(tabixIdxFile.exists() && tabixIdxFile.length() > 0);
      System.err.println("Catalog [" + buildInfo.getCatalogPrefix() + "] last modified timestamp: " +
         catOutFile.lastModified() +
         " Date: " + new SimpleDateFormat("MM.dd.yyyy 'at' HH:mm:ss z").format(catOutFile.lastModified()));
      System.err.println("Tabix last modified timestamp: " +
         tabixIdxFile.lastModified() +
         " Date: " + new SimpleDateFormat("MM.dd.yyyy 'at' HH:mm:ss z").format(tabixIdxFile.lastModified()));
      assertTrue(tabixIdxFile.lastModified() >= catOutFile.lastModified());
   }

   @Test
   public void testUnsortedCol7JSON() throws IOException, BuildCatalogStepInputException
   {
      String catalogPrefix = "createStepTest_unsortedCatalog";
      ensureCatalogDeleted(catalogPrefix);

      mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), catalogPrefix);  // TJSON input file
      BuildInfo buildInfo = new BuildInfo(mBuildInfoMap);
      buildInfo.setInputJSONColumnNum(7); // lets not default to -1.
      buildInfo.setSortCatalogOnCreate("false"); // default is to sort so lets not sort the catalog yet.
      try
      {
         CreateStep createCatalogStep = new CreateStep(buildInfo, new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), new File(mTargetDirName)));
         createCatalogStep.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         //System.err.println("input exception occurred: " + inputE);
         fail("Unexpected exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         //System.err.println("exec exception occurred: " + execE);
         fail("Unexpected exception occurred: " + execE);
      }

      String catalogOutFilePath = mTargetDirName + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
      String catalogTabixIndexPath = catalogOutFilePath + ".tbi";

      File catOutFile = new File(catalogOutFilePath);
      File tabixIdxFile = new File(catalogTabixIndexPath);

      assertTrue(catOutFile.exists() && catOutFile.length() > 0);
      assertTrue(tabixIdxFile.exists() && tabixIdxFile.length() > 0);

      System.err.println("Catalog [" + buildInfo.getCatalogPrefix() + "] last modified timestamp: " +
         catOutFile.lastModified() +
         " Date: " + new SimpleDateFormat("MM.dd.yyyy 'at' HH:mm:ss z").format(catOutFile.lastModified()));
      System.err.println("Tabix last modified timestamp: " +
         tabixIdxFile.lastModified() +
         " Date: " + new SimpleDateFormat("MM.dd.yyyy 'at' HH:mm:ss z").format(tabixIdxFile.lastModified()));
      assertTrue(tabixIdxFile.lastModified() >= catOutFile.lastModified());
   }

   @Test
   public void testBadJSONCol() throws IOException, BuildCatalogStepInputException
   {
      String catalogPrefix = "createStepTest_badJsonColCatalog";
      ensureCatalogDeleted(catalogPrefix);

      mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), "x");
      BuildInfo buildInfo = new BuildInfo(mBuildInfoMap);
      buildInfo.setInputJSONColumnNum(3); // lets not default to -1 yet.
      buildInfo.setSortCatalogOnCreate("false"); // default is to sort so lets not sort the catalog yet.

      try
      {
         CreateStep createCatalogStep = new CreateStep(buildInfo, new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), new File(mTargetDirName)));
         createCatalogStep.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         System.out.println("We correctly got an input exception b/c of the input json column value of 3. This is not valid JSON input column data. Exception Message: " + inputE.getMessage());
         assertTrue(true);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         fail("We should not have gotten to an execution exception. Should have failed in input validation.");
      }

      String catalogOutFilePath = mTargetDirName + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
      String catalogTabixIndexPath = catalogOutFilePath + ".tbi";

      File catOutFile = new File(catalogOutFilePath);
      File tabixIdxFile = new File(catalogTabixIndexPath);
      assertTrue(!catOutFile.exists());
      assertTrue(!tabixIdxFile.exists());

      //  Now try a bogus column # altogether:
      buildInfo.setInputJSONColumnNum(8); // lets not default to -1 yet.
      try
      {
         CreateStep createCatalogStep2 = new CreateStep(buildInfo, new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), new File(mTargetDirName)));
         createCatalogStep2.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         System.out.println("We correctly got an input exception b/c of the input json column value. This is not valid input column number. Exception Message: " + inputE.getMessage());
         assertTrue(true);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         fail("We should not have gotten to an execution exception. Should have failed in input validation.");
      }

      catalogOutFilePath = mTargetDirName + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
      catalogTabixIndexPath = catalogOutFilePath + ".tbi";

      catOutFile = new File(catalogOutFilePath);
      tabixIdxFile = new File(catalogTabixIndexPath);
      assertTrue(!catOutFile.exists());
      assertTrue(!tabixIdxFile.exists());
   }

   @Test
   public void testSortedByUserList() throws IOException, BuildCatalogStepInputException
   {
      String catalogPrefix = "createStepTest_sortedByUserList";
      ensureCatalogDeleted(catalogPrefix);

      mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), catalogPrefix);  // TJSON input file
      BuildInfo buildInfo = new BuildInfo(mBuildInfoMap);
      buildInfo.setChromosomeSortOrderFileName(reverseChrSortOrderFile);  // we could do some sort of reverse-order test is easy.

      try
      {
         CreateStep createCatalogStep = new CreateStep(buildInfo, new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), new File(mTargetDirName)));
         createCatalogStep.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         fail("Unexpected input exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         fail("Unexpected exec exception occurred: " + execE);
      }

      String catalogOutFilePath = mTargetDirName + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
      String catalogTabixIndexPath = catalogOutFilePath + ".tbi";

      File catOutFile = new File(catalogOutFilePath);
      File tabixIdxFile = new File(catalogTabixIndexPath);

      assertTrue(catOutFile.exists() && catOutFile.length() > 0);
      assertTrue(tabixIdxFile.exists() && tabixIdxFile.length() > 0);

      System.err.println("Catalog [" + buildInfo.getCatalogPrefix() + "] last modified timestamp: " +
         catOutFile.lastModified() +
         " Date: " + new SimpleDateFormat("MM.dd.yyyy 'at' HH:mm:ss z").format(catOutFile.lastModified()));
      System.err.println("Tabix last modified timestamp: " +
         tabixIdxFile.lastModified() +
         " Date: " + new SimpleDateFormat("MM.dd.yyyy 'at' HH:mm:ss z").format(tabixIdxFile.lastModified()));
      assertTrue(tabixIdxFile.lastModified() >= catOutFile.lastModified());
   }

   private void ensureCatalogDeleted(String catalogFilePrefix)
   {
      File catOutFile = new File(catalogFilePrefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      File tabixOutFile = new File(catalogFilePrefix + CatalogMetadataConstant.CATALOG_TABIX_INDEX_SUFFIX);
      if (catOutFile.exists())
      {
         if (catOutFile.delete())
         {
            try
            {
               Thread.sleep(300);
            }
            catch (InterruptedException i)
            {/* fine, just ignore */}
            if (!catOutFile.exists())
            {
               // we are good.
            }
         }
         else
         {
            fail("Catalog file was not removed before test.");
         }
      }
      if (tabixOutFile.exists())
      {
         if (tabixOutFile.delete())
         {
            try
            {
               Thread.sleep(300);
            }
            catch (InterruptedException i)
            {/* fine, just ignore */}
            if (!tabixOutFile.exists())
            {
               // we are good.
            }
         }
         else
         {
            fail("Catalog tabix file was not removed before test.");
         }
      }
   }
}
