package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.bior.cli.cmd.BuildCatalogCommand;
import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;
import edu.mayo.pipes.util.BiorProperties;

public class BuildCatalogStepTest  extends BaseFunctionalTest
{
   private static final String NL = System.getProperty("line.separator");

   private File mTargetDir = null;
   private File mTempDir = null;

   private Map<String, String> mBuildInfoMap = new HashMap<String, String>();
   private BuildInfo mBuildInfo = null;
   private File mCatalogOutFile = null;
   private File mCatalogTabixIndex = null;

   private File mBuildInfoFile = null;


   
   @Before
   public void beforeEach() throws Exception
   {
      TemporaryFolder tempFolder = new TemporaryFolder();
      tempFolder.create();
      mTargetDir = tempFolder.newFolder("BuildCatalogStepTest_TARGET");
      mTempDir = tempFolder.newFolder("BuildCatalogStepTest_TEMP");

      String catalogPrefix = "buildCatalog.allSteps";

      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), 		HumanBuildAssembly.GRCh37.name());
      mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), 		catalogPrefix);
      mBuildInfoMap.put(BuildInfoKey.INDEXES.name(), 				"key, bbb");
      mBuildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), 	"");
      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(), 			"dbSNP");
      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), 	"142");
      mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_RELEASE_DATE.name(),"2018-07-10");
      mBuildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), 			mTargetDir.getCanonicalPath());
      mBuildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), 				mTempDir.getCanonicalPath());
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), 		"");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), 	new File("src/test/resources/buildCatalog/makeJsonStep.sh").getCanonicalPath());

      mBuildInfoFile = new File(mTempDir, "build_info.txt");

      mCatalogOutFile = new File(mTargetDir, catalogPrefix + ".tsv.bgz");
      mCatalogTabixIndex = new File(mTargetDir, mCatalogOutFile.getName() + ".tbi");
   }


   /**
    * Test all steps in the catalog build process.  At the end, we should have:
    * TARGET_DIRECTORY/
    *   buildCatalog.allSteps.tsv.bgz
    *   buildCatalog.allSteps.tsv.bgz.tbi
    *   buildCatalog.allSteps.columns.tsv
    *   buildCatalog.allSteps.columns.tsv.blacklist
    *   buildCatalog.allSteps.columns.tsv.blacklist.biorweb
    *   buildCatalog.allSteps.datasource.properties
    *   index/
    *     buildCatalog.allSteps.rsID???.idx.h2.db
    *     buildCatalog.allSteps.CHR????.idx.h2.db
    *   build/
    *     buildCatalog.allSteps.columns.tsv.default
    *     buildCatalog.allSteps.datasource.properties.default
    *   progress/
    *     CREATE_CATALOG.log
    *     CREATE_CATALOG.summary
    *     CREATE_CATALOG.successful
    *     MAKE_JSON.log
    *     MAKE_JSON.summary
    *     MAKE_JSON.successful
    *     MERGE_PROP_FILES.log
    *     MERGE_PROP_FILES.summary
    *     MERGE_PROP_FILES.succesful
    *     PROP_FILES.log
    *     PROP_FILES.summary
    *     PROP_FILES.successful
    *     INDEXES.log
    *     INDEXES.summary
    *     INDEXES.successful
    *     VERIFY_CATALOG.log
    *     VERIFY_CATALOG.summary
    *     VERIFY_CATALOG.successful
    *     HELP_DOCS.log
    *     HELP_DOCS.summary
    *     HELP_DOCS.successful
    *     SUMMARY.log
    *     SUMMARY.summary
    *     SUMMARY.successful
    */
   @Test
   public void testAllSteps() throws IOException, Exception
   {
	  BiorProperties.setFile("src/test/resources/bior.properties.test");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile  inputFile  --column  4");
      File compileOutput = new File(mTempDir, "compileOutput.tsv");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), compileOutput.getCanonicalPath());
      mBuildInfo = new BuildInfo(mBuildInfoMap);
      mBuildInfo.save(mBuildInfoFile.getCanonicalPath());

      BuildCatalog build = new BuildCatalog();
      build.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.AUTO);

      assertDirContentsAfterAllSteps();
    }

   
   /**
    * Test the start and stop flags.
    * First, build the catalog and properties files.
    *     TARGET_DIRECTORY/
    *       buildCatalog.allSteps.tsv.bgz
    *       buildCatalog.allSteps.tsv.bgz.tbi
    *       buildCatalog.allSteps.columns.tsv
    *       buildCatalog.allSteps.columns.tsv.blacklist
    *       buildCatalog.allSteps.columns.tsv.blacklist.biorweb
    *       buildCatalog.allSteps.datasource.properties
    *     build/
    *       buildCatalog.allSteps.columns.tsv.default
    *       buildCatalog.allSteps.datasource.properties.default
    *     progress/
    *       CREATE_CATALOG.log
    *       CREATE_CATALOG.summary
    *       CREATE_CATALOG.successful
    *       MAKE_JSON.log
    *       MAKE_JSON.summary
    *       MAKE_JSON.successful
    *       PROP_FILES.log
    *       PROP_FILES.summary
    *       PROP_FILES.successful
    *       MERGE_PROP_FILES.log
    *       MERGE_PROP_FILES.summary
    *       MERGE_PROP_FILES.successful

    * Then create tabix indexes
    *     TARGET_DIRECTORY/
    *       index/
    *         buildCatalog.allSteps.rsID???.idx.h2.db
    *         buildCatalog.allSteps.CHR????.idx.h2.db
    *       progress/
    *         INDEXES.log
    *         INDEXES.summary
    *         INDEXES.successful

    * Then verify and STOP
    *     TARGET_DIRECTORY/
    *       progress/
    *         VERIFY_CATALOG.log
    *         VERIFY_CATALOG.summary
    *         VERIFY_CATALOG.successful
    *        
    * The progress dir should NOT contain these files:
    *     TARGET_DIRECTORY/
    *       progress/
    *         HELP_DOCS.log
    *         HELP_DOCS.summary
    *         HELP_DOCS.successful
    *         SUMMARY.log
    *         SUMMARY.summary
    *         SUMMARY.successful
    */
   @Test
   public void testStartAndStopSteps() throws IOException, Exception
   {
	  BiorProperties.setFile("src/test/resources/bior.properties.test");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile  inputFile  --column  4");
      File compileOutput = new File(mTempDir, "compileOutput.tsv");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), compileOutput.getCanonicalPath());
      mBuildInfo = new BuildInfo(mBuildInfoMap);
      mBuildInfo.save(mBuildInfoFile.getCanonicalPath());

      BuildCatalog build = new BuildCatalog();
      
      // First build with the first 3 steps:  MAKE_JSON, MAKE_CATALOG, MAKE_PROP_FILES, MERGE_PROP_FILES 
      build.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.MAKE_JSON, BuildStepKey.MERGE_PROP_FILES);
      assertFilesThruStep(BuildStepKey.MERGE_PROP_FILES);

      // Then create the h2 db indexes:  MAKE_JSON, MAKE_CATALOG, MAKE_PROP_FILES, MERGE_PROP_FILES 
      build.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.MAKE_INDEXES, BuildStepKey.MAKE_INDEXES);
      assertFilesThruStep(BuildStepKey.MAKE_INDEXES);
      
      // Then run Verify:  
      build.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.VERIFY, BuildStepKey.VERIFY);
      assertFilesThruStep(BuildStepKey.VERIFY);
   }
   
   @Test
   public void testStartAndStopSteps_withCmd() throws IOException, Exception
   {
	  BiorProperties.setFile("src/test/resources/bior.properties.test");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile  inputFile  --column  4");
      File compileOutput = new File(mTempDir, "compileOutput.tsv");
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), compileOutput.getCanonicalPath());
      mBuildInfo = new BuildInfo(mBuildInfoMap);
      mBuildInfo.save(mBuildInfoFile.getCanonicalPath());

      BuildCatalog build = new BuildCatalog();
      
      // First build with the first 3 steps:  MAKE_JSON, MAKE_CATALOG, MAKE_PROP_FILES, MERGE_PROP_FILES
      CommandOutput out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog", "--buildInfo", mBuildInfoFile.getCanonicalPath(),  "--startStep", "make_json",  "--endStep", "merge_prop_files");
      assertNoErrorsOrWarnings(out);
      assertFilesThruStep(BuildStepKey.MERGE_PROP_FILES);
      assertContainsAll(out.stdout, Arrays.asList(
    		  "Step make_json SUCCEEDED",
    		  "Step make_catalog SUCCEEDED",
    		  "Step make_prop_files SUCCEEDED",
    		  "Step merge_prop_files SUCCEEDED"
    		  ));


      // Then create the h2 db indexes:  MAKE_JSON, MAKE_CATALOG, MAKE_PROP_FILES, MERGE_PROP_FILES 
      out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog", "--buildInfo", mBuildInfoFile.getCanonicalPath(),  "--startStep", "make_indexes",  "--endStep", "make_indexes");
      assertNoErrorsOrWarnings(out);
      assertFilesThruStep(BuildStepKey.MAKE_INDEXES);
      assertContainsAll(out.stdout, Arrays.asList(
    		  "Creating index on key 'key', path",
    		  "Creating index on key 'bbb', path"
    		  ));
      
      // Then run Verify:  
      out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog", "--buildInfo", mBuildInfoFile.getCanonicalPath(),  "--startStep", "verify",  "--endStep", "verify");
      assertNoErrorsOrWarnings(out);
      assertFilesThruStep(BuildStepKey.VERIFY);
      assertContains(out.stdout, "Step verify SUCCEEDED");
   }

   	private void assertNoErrorsOrWarnings(CommandOutput out) {
   		assertEquals(/*msg=*/out.stderr, 0, out.exit);
   		assertTrue(/*msg=*/out.stdout, out.stdout.contains("Variables from build_info.txt"));

   		assertFalse(/*msg=*/out.stdout, out.stdout.contains("Unexpected error executing step"));
   		assertFalse(/*msg=*/out.stderr, out.stderr.contains("Unexpected error executing step"));
   	}


   private void assertFilesThruStep(BuildStepKey step) throws IOException {
	   List<String> filesInTargetDir = new ArrayList<String>(Arrays.asList(
  	         BuildCatalog.BUILD_SUBDIR,   // DIR
  	         "buildCatalog.allSteps.columns.tsv",
  	         "buildCatalog.allSteps.columns.tsv.blacklist",
  	         "buildCatalog.allSteps.columns.tsv.blacklist.biorweb",
  	         "buildCatalog.allSteps.datasource.properties",
  	         "buildCatalog.allSteps.tsv.bgz",
  	         "buildCatalog.allSteps.tsv.bgz.tbi"
  	      ));
	   // NOTE: The index/ dir does not exist yet until an index is built:
	   if( step.ordinal() >= BuildStepKey.MAKE_INDEXES.ordinal()) {
		   filesInTargetDir.add("index");      // DIR
		   MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, "index"), Arrays.asList(
		    		  	"buildCatalog.allSteps.bbb.idx.h2.db",
		    		  	"buildCatalog.allSteps.key.idx.h2.db"
		    		  	));
	   }
	   MergeColumnsTsvTest.assertDirContainsOnly(mTargetDir, filesInTargetDir);
	   
	   
	   MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR), Arrays.asList(
	    	         "buildCatalog.allSteps.columns.tsv.blacklist.biorweb.default",
	    	         "buildCatalog.allSteps.columns.tsv.blacklist.default",
	    	         "buildCatalog.allSteps.columns.tsv.default",
	    	         "buildCatalog.allSteps.datasource.properties.01",	// Backup of the datasource.properties file since it is modified
	    	         "buildCatalog.allSteps.datasource.properties.default",
	    	         "progress"
	    	      ));
	   
	   
	   List<String> filesInProgressDir = new ArrayList<String>(Arrays.asList(
			 "make_json.log",
			 "make_json.successful",
			 "make_json.summary"
			 ));
	   if( step.ordinal() >= BuildStepKey.MAKE_CATALOG.ordinal()) {
		   filesInProgressDir.addAll( Arrays.asList(
  	         "make_catalog.log",
  	         "make_catalog.successful",
  	         "make_catalog.summary"
  	         ));
	   }
	   if( step.ordinal() >= BuildStepKey.MAKE_PROP_FILES.ordinal()) {
		   filesInProgressDir.addAll( Arrays.asList(
  	         "make_prop_files.log",
  	         "make_prop_files.successful",
  	         "make_prop_files.summary"
  	         ));
	   }
	   if( step.ordinal() >= BuildStepKey.MERGE_PROP_FILES.ordinal()) {
		   filesInProgressDir.addAll( Arrays.asList(
  	         "merge_prop_files.log",
  	         "merge_prop_files.successful",
  	         "merge_prop_files.summary"
  	         ));
	   }
	   if( step.ordinal() >= BuildStepKey.MAKE_INDEXES.ordinal()) {
		   filesInProgressDir.addAll( Arrays.asList(
			 "make_indexes.log",
  	         "make_indexes.successful",
  	         "make_indexes.summary"
  	         ));
	   }
	   if( step.ordinal() >= BuildStepKey.VERIFY.ordinal()) {
		   filesInProgressDir.addAll( Arrays.asList(
		  	 "verify.log",
		  	 "verify.successful",
		  	 "verify.summary"
		  	 ));
	   }
  	         //"summary.log",
  	         //"summary.successful",
  	         //"summary.summary",
  	         //"docs.log",
  	         //"docs.successful",
  	         //"docs.summary",
	   MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress"), filesInProgressDir);
   }


   @Test
   /** When the columns.tsv, datasource.properties, and blacklist files are modified, they should not be replace */
   public void testAllSteps_filesModifiedAndRerun() throws IOException, Exception
   {
	   testAllSteps();
	   
	   // Change columns.tsv, datasource.properties, blacklist files
	   String targetDir = mBuildInfoMap.get(BuildInfoKey.TARGET_DIR.name());
	   String ctgPrefix = mBuildInfoMap.get(BuildInfoKey.CATALOG_PREFIX.name());
	   File columnsTsv   = new File(targetDir, ctgPrefix + ".columns.tsv");
	   File datasrcProps = new File(targetDir, ctgPrefix + ".datasource.properties");
	   File blacklist    = new File(targetDir, ctgPrefix + ".columns.tsv.blacklist");
	   File blacklistBiorweb = new File(targetDir, ctgPrefix + ".columns.tsv.blacklist.biorweb");
	   
	   // Modify and save columns.tsv  (add extra line)
	   String columnsTsvStr = FileUtils.readFileToString(columnsTsv) + concat("TestCol", "String", ".", "MiniDesc", "Test Column") + "\n";
	   FileUtils.writeStringToFile(columnsTsv, columnsTsvStr);
	   
	   // Modify and save datasource.properties
	   String datasourceStr = FileUtils.readFileToString(datasrcProps) + "## Extra field" + "\n" + "ExtraField=nothing" + "\n";
	   FileUtils.writeStringToFile(datasrcProps, datasourceStr);
	   
	   // Modify and save blacklist
	   String blacklistStr = "_landmark" + "\n";
	   FileUtils.writeStringToFile(blacklist, blacklistStr);
	   
	   // Modify and save blacklist.biorweb
	   String blacklistBiorwebStr = "_minBP" + "\n" + "_maxBP" + "\n";
	   FileUtils.writeStringToFile(blacklistBiorweb, blacklistBiorwebStr);
	   
	   // Build again
	   BuildCatalog build = new BuildCatalog();
	   build.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.AUTO);
	   
	   // Verify columns.tsv, datasource.properties, and blacklist files are not overwritten
	   assertEquals(columnsTsvStr, 		FileUtils.readFileToString(columnsTsv));
	   assertEquals(datasourceStr, 		FileUtils.readFileToString(datasrcProps));
	   assertEquals(blacklistStr, 		FileUtils.readFileToString(blacklist));
	   assertEquals(blacklistBiorwebStr,FileUtils.readFileToString(blacklistBiorweb));
   }


   
   private void assertDirContentsAfterAllSteps() throws IOException
   {
      MergeColumnsTsvTest.assertDirContainsOnly(mTargetDir, Arrays.asList(
    	 BuildCatalog.BUILD_SUBDIR,   // DIR
         "buildCatalog.allSteps.columns.tsv",
         "buildCatalog.allSteps.columns.tsv.blacklist",
         "buildCatalog.allSteps.columns.tsv.blacklist.biorweb",
         "buildCatalog.allSteps.datasource.properties",
         "buildCatalog.allSteps.tsv.bgz",
         "buildCatalog.allSteps.tsv.bgz.tbi",
         "index"      // DIR
      ));
      MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, "index"), Arrays.asList(
         "buildCatalog.allSteps.bbb.idx.h2.db",
         "buildCatalog.allSteps.key.idx.h2.db"
      ));
      MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR), Arrays.asList(
         "buildCatalog.allSteps.columns.tsv.blacklist.biorweb.default",
         "buildCatalog.allSteps.columns.tsv.blacklist.default",
         "buildCatalog.allSteps.columns.tsv.default",
         "buildCatalog.allSteps.datasource.properties.01",	// Backup of the datasource.properties file since it is modified
         "buildCatalog.allSteps.datasource.properties.default",
         "progress"
      ));
      MergeColumnsTsvTest.assertDirContainsOnly(new File(mTargetDir, BuildCatalog.BUILD_SUBDIR + "/progress"), Arrays.asList(
         //"docs.log",
         //"docs.successful",
         //"docs.summary",
         "make_catalog.log",
         "make_catalog.successful",
         "make_catalog.summary",
         "make_indexes.log",
         "make_indexes.successful",
         "make_indexes.summary",
         "make_json.log",
         "make_json.successful",
         "make_json.summary",
         "make_prop_files.log",
         "make_prop_files.successful",
         "make_prop_files.summary",
         "merge_prop_files.log",
         "merge_prop_files.successful",
         "merge_prop_files.summary",
         //"summary.log",
         //"summary.successful",
         //"summary.summary",
         "verify.log",
         "verify.successful",
         "verify.summary"
      ));
   }

   @Test
   public void testSomeSteps() throws IOException, BuildCatalogStepInputException
   {
      // TJSON input file to the create step.  NOTE: Do NOT use if running the compile step as it will overwrite this file
      File inputToCreateStep = new File(mTempDir, "tabToTjsonOutput.tsv");
      FileUtils.copyFile(new File("src/test/resources/createCatalog/tabToTjsonOutput.tsv"), inputToCreateStep);
      mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), inputToCreateStep.getCanonicalPath());
      mBuildInfo = new BuildInfo(mBuildInfoMap);
      mBuildInfo.save(mBuildInfoFile.getCanonicalPath());


      stepCatalogCreation();
      stepPropFileCreation();
      stepIndexCatalog();
      stepVerifyCatalog();
   }

   @Test
   // Test each of the start steps to make sure they match the expected
   public void testAllStartStepNames() throws IOException, Exception {
	   mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile  inputFile  --column  4");
	   mBuildInfo = new BuildInfo(mBuildInfoMap);
	   mBuildInfo.save(mBuildInfoFile.getCanonicalPath());

	   BuildStepKey[] STEPS = new BuildStepKey[] {
			   BuildStepKey.AUTO,
			   BuildStepKey.MAKE_JSON,
			   BuildStepKey.MAKE_CATALOG,
			   BuildStepKey.MAKE_INDEXES,
			   BuildStepKey.MERGE_PROP_FILES,
			   BuildStepKey.MAKE_PROP_FILES,
			   BuildStepKey.VERIFY
	   };
	   
	   for(BuildStepKey step : STEPS) {
		   BuildCatalog build = new BuildCatalog();
		   build.build(mBuildInfoFile.getCanonicalPath(), step);
	   }
	   
	   // Also test case-insensitive
	   
	   // Also try one that is not expected
   }
   
   
   //========================================================================================================================================
   
   private void stepCatalogCreation()
   {
      try
      {
         CreateStep createCatalogStep = new CreateStep(mBuildInfo, new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), mTargetDir));
         createCatalogStep.execute();
         System.out.println("Catalog created: " + mCatalogOutFile.getCanonicalPath());
      }
      catch (IOException io)
      {
         System.err.println("IO Exception occurred: " + io.getMessage());
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         System.err.println("Input Exception occurred: " + inputE.getMessage());
         fail("Unexpected exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         System.err.println("Execution Exception occurred: " + execE.getMessage());
         fail("Unexpected exception occurred: " + execE);
      }

      assertTrue(mCatalogOutFile.exists() && mCatalogOutFile.length() > 0);
      assertTrue(mCatalogTabixIndex.exists() && mCatalogTabixIndex.length() > 0);
      assertTrue(mCatalogTabixIndex.lastModified() >= mCatalogOutFile.lastModified());
   }

   private void stepPropFileCreation() throws IOException
   {
      assertTrue(mCatalogOutFile.exists() && mCatalogOutFile.length() > 0);

      ///////////////////////////////////////////////////
      // Ok, we have a catalog. lets create the prop files,
      // which is the real thing testing in this class.
      //////////////////////////////////////////////////
      try
      {
         CreatePropFilesStep createCatalogPropFiles = new CreatePropFilesStep(mBuildInfo, new StepLogger(BuildStepKey.MAKE_PROP_FILES.toString(), mTargetDir));
         createCatalogPropFiles.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         inputE.printStackTrace();
         fail("Unexpected exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         execE.printStackTrace();
         fail("Unexpected exception occurred: " + execE);
      }

      // List of files under "target", "target/[BuildCatalog.BUILD_SUBDIR]", and "target/[BuildCatalog.BUILD_SUBDIR]/progress" should match:
      String colsTsvActual = FileUtils.readFileToString(new File(mTargetDir, mBuildInfo.getCatalogPrefix() + CatalogMetadataConstant.COLUMN_INFO_SUFFIX));
      String colsTsvBlacklistActual = FileUtils.readFileToString(new File(mTargetDir, mBuildInfo.getCatalogPrefix() + CatalogMetadataConstant.BLACKLIST_SUFFIX));
      String colsTsvBlacklistBiorwebActual = FileUtils.readFileToString(new File(mTargetDir, mBuildInfo.getCatalogPrefix() + CatalogMetadataConstant.BLACKLIST_BIORWEB_SUFFIX));
      String datasrcActual = FileUtils.readFileToString(new File(mTargetDir, mBuildInfo.getCatalogPrefix() + CatalogMetadataConstant.DATASRC_PROPS_SUFFIX));
      // For easy comparison, we have to convert the TEMP_DIR to "/TEMP_DIR" in these two cases (one with the additional "tmp.....bior_build_catalog" subdir):
      //   /private/var/folders/gl/wdqzn5ws0jz5qtzy41x7h0d9ygjpzy/T/junit3856436223634400083/BuildCatalogStepTest_TEMP
      //   /private/var/folders/gl/wdqzn5ws0jz5qtzy41x7h0d9ygjpzy/T/junit3856436223634400083/BuildCatalogStepTest_TEMP/tmp7046163851327730055bior_build_catalog
      String logActual = MergeColumnsTsvTest.removeTimestamp(
         FileUtils.readFileToString(new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildStepKey.MAKE_CATALOG.toString() + ".log")))
	     .replaceAll(mTempDir.getCanonicalPath() + "/tmp.*bior_build_catalog", "/TEMP_DIR")
	     .replaceAll(mTempDir.getCanonicalPath(), "/TEMP_DIR")
	     .replaceAll(mTargetDir.getCanonicalPath(), "/TARGET_DIR");
      String summaryActual = FileUtils.readFileToString(new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildStepKey.MAKE_CATALOG.toString() + ".summary"));
      ;
      // The successful file is added by the full build process
      //String successfulActual				= FileUtils.readFileToString(new File(mTargetDir + "/" + BuildCatalog.BUILD_SUBDIR + "/progress", BuildCatalog.BuildStepKey.CREATE_CATALOG.toString() + ".successful"));;

      String logExpected =
         "CreateStep.init(): TEMP_DIR value: /TEMP_DIR" + NL +
         "make_catalog step: Checking inputs..." + NL +
            "CreateStep.checkInput(): Creating TEMP_DIR value: /TEMP_DIR" + NL +
            "Catalog file path we will be creating: /TARGET_DIR/buildCatalog.allSteps.tsv.bgz" + NL +
            "Creating catalog from JSON in last column of: '/TEMP_DIR/tabToTjsonOutput.tsv'" + NL +
            "Catalog created: /TARGET_DIR/buildCatalog.allSteps.tsv.bgz" + NL +
            "Tabix index created: /TARGET_DIR/buildCatalog.allSteps.tsv.bgz.tbi" + NL +
            "Create catalog step: checking outputs ... /TEMP_DIR/tabToTjsonOutput.tsv" + NL;

      String dirForExpectedProgress = "src/test/resources/buildCatalog/propertiesFilesExpected/build/progress";
      String dirForExpectedFiles = "src/test/resources/buildCatalog/mergeProps";
      String colsTsvExpected = FileUtils.readFileToString(new File(dirForExpectedFiles, "x.columns.tsv"));
      String colsTsvBlacklistExpected = FileUtils.readFileToString(new File(dirForExpectedFiles, "x.columns.tsv.blacklist"));
      // This one has two additional fields in it that won't appear in the first run-thru  ("ALTS", "POS")
      String colsTsvBlacklistBiorwebExpected = FileUtils.readFileToString(new File(dirForExpectedFiles, "x.columns.tsv.blacklist.biorweb"))
    		  .replace("ALTS\nPOS", "");
      String datasrcExpected = FileUtils.readFileToString(new File(dirForExpectedFiles, "x.datasource.properties")).replace("for Catalog - x", "for Catalog - buildCatalog.allSteps");


      assertEquals(colsTsvExpected, colsTsvActual);
      assertEquals(colsTsvBlacklistExpected, colsTsvBlacklistActual);
      assertEquals(colsTsvBlacklistBiorwebExpected, colsTsvBlacklistBiorwebActual);
      assertEquals(datasrcExpected, datasrcActual);
      assertEquals(logExpected, logActual);
      assertContainsAll(summaryActual, Arrays.asList(
    		  "Creating catalog from JSON in last column of",
    		  "Catalog created",
    		  "Tabix index created"
    		  ));
   }

   private void stepIndexCatalog() throws BuildCatalogStepInputException
   {
      assertTrue(mCatalogOutFile.exists() && mCatalogOutFile.length() > 0);

      List<String> indexes = Arrays.asList(new String[]{"RSID", "GENENAME"});
      mBuildInfoMap.put(BuildInfoKey.INDEXES.name(), StringUtils.join(indexes, ","));
      mBuildInfo = new BuildInfo(mBuildInfoMap);

      ///////////////////////////////////////////////////
      // Ok, we have a catalog. lets run the verify catalog step:
      //////////////////////////////////////////////////
      try
      {
         IndexesStep indexStep = new IndexesStep(mBuildInfo, new StepLogger(BuildStepKey.MAKE_INDEXES.toString(), mTargetDir));
         indexStep.execute();

         File geneNameIdx = new File(mTargetDir, "index/buildCatalog.allSteps.GENENAME.idx.h2.db");
         File rsidIdx = new File(mTargetDir, "index/buildCatalog.allSteps.RSID.idx.h2.db");
         assertTrue(geneNameIdx.exists() && geneNameIdx.length() > 0);
         assertTrue(rsidIdx.exists() && rsidIdx.length() > 0);
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         inputE.printStackTrace();
         fail("Unexpected input exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         execE.printStackTrace();
         fail("Unexpected execution exception occurred: " + execE);
      }
   }


   private void stepVerifyCatalog()
   {
      assertTrue(mCatalogOutFile.exists() && mCatalogOutFile.length() > 0);

      ///////////////////////////////////////////////////
      // Ok, we have a catalog. lets run the verify catalog step:
      //////////////////////////////////////////////////
      try
      {
         VerifyStep verifyStep = new VerifyStep(mBuildInfo, new StepLogger(BuildStepKey.VERIFY.toString(), mTargetDir));
         verifyStep.execute();
      }
      catch (IOException io)
      {
         io.printStackTrace();
         fail("Unexpected io exception occurred: " + io);
      }
      catch (BuildCatalogStepInputException inputE)
      {
         inputE.printStackTrace();
         fail("Unexpected input exception occurred: " + inputE);
      }
      catch (BuildCatalogStepExecuteException execE)
      {
         execE.printStackTrace();
         fail("Unexpected execution exception occurred: " + execE);
      }
   }


}
