/*
 * BioR Web
 * 
 * Created: Dec 22, 2010
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2010
 * 
 */

package edu.mayo.bior.util;

import java.util.Collection;


/**
 * Class that handles the keyword information
 * 
 * @author Greg Dougherty
 */
public class Keyword 
{
	/** The Unique name of the Keyword, to use in lookups */
	private String	uniqueName;
	/** The human used name of the keyword, will only be unique within a Category */
	private String	description;
	/** The metadata for the keyword, to use as hover text */
	private String	metadata;
	
	
	/**
	 * Only to be used by IsSerializable
	 */
	protected Keyword ()
	{
		super ();
		uniqueName = description = metadata = null;
	}
	
	
	/**
	 * Constructor
	 * 
	 * @param uniqueName	The Unique name of the Keyword, to use in lookups
	 * @param description	The human used name of the keyword, will only be unique within a Category
	 * @param metadata		The metadata for the keyword, to use as hover text
	 */
	public Keyword (String uniqueName, String description, String metadata)
	{
		this.uniqueName = uniqueName;
		this.description = description;
		this.metadata = metadata;
	}
	
	
	/**
	 * Look through a collection of Keywords, finding the first one whose uniqueName matches key
	 * 
	 * @param theKeywords	Collection to look through
	 * @param key			Name to look for
	 * @return	The first matching Keyword, or null if didn't find a match
	 */
	public static final Keyword getFromList (Collection<Keyword> theKeywords, String key)
	{
		if (key == null)
			return null;
		
		for (Keyword theKeyword : theKeywords)
		{
			if (theKeyword.uniqueName.equalsIgnoreCase (key))
				return theKeyword;
		}
		
		return null;
	}
	
	
	/**
	 * @return the uniqueName
	 */
	public final String getName ()
	{
		return uniqueName;
	}
	
	
	/**
	 * @return the description
	 */
	public final String getDescription ()
	{
		return description;
	}
	
	
	/**
	 * @return the metadata
	 */
	public final String getMetadata ()
	{
		return metadata;
	}
	
	
	/**
	 * Two "equal" Objects must have the same hash code.  If description is null in either of the Keyword
	 * Objects, then we only compare on uniqueName.  So description can only be part of the hash code if 
	 * uniqueName is null
	 */
	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode ()
	{
		final int prime = 31;
		int result = 1;
		result = prime * result + ((uniqueName == null) ? 0 : uniqueName.toLowerCase ().hashCode ());
		if (uniqueName == null)
			result = prime * result + ((description == null) ? 0 : description.toLowerCase ().hashCode ());
		return result;
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals (Object obj)
	{
		if (this == obj)
			return true;
		
		if (obj == null)
			return false;
		
		if ((obj instanceof String))
			return compareValue ((String) obj, uniqueName);
		
		if (!(obj instanceof Keyword))
			return false;
		
		Keyword other = (Keyword) obj;
		if ((description == null) || (other.description == null))
			return compareValue (other.uniqueName, uniqueName);
		
		if (uniqueName == null)
		{
			if (other.uniqueName != null)
				return false;
		}
		else if (!uniqueName.equalsIgnoreCase (other.uniqueName))
			return false;
		
//		if (description == null)	// if description was null, would not have made it here
//		{
//			if (other.description != null)
//				return false;
//		}
//		else 
		if (!description.equalsIgnoreCase (other.description))
			return false;
		
		return true;
	}
	
	
	/**
	 * Compare two strings
	 * 
	 * @param test	Foreign string
	 * @param value	Local string
	 * @return	True if both are null, or if equalsIgnoreCase returns true
	 */
	private static final boolean compareValue (String test, String value)
	{
		if (value == null)
			return (test == null);
		
		return (value.equalsIgnoreCase (test));
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString ()
	{
		return "Keyword [" + uniqueName + " : " + description + " : " + metadata + "]";
	}
	
}
