/**
 * BioRWeb
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.util;

import java.util.*;

/**
 * Class that handles passing category information back and forth
 *
 * <p>@author Gregory Dougherty</p>
 */
public class Category 
{
	/** The human used name of the Category, will only be unique at its level of the tree */
	private String			name;
	/** The Unique name of the Category, to use in lookups */
	private String			uniqueName;
	/** The metadata for the Category, to use as hover text */
	private String			metadata;
	private Category		parent;
	private List<Keyword>	keywords;
	
	
	/**
	 * Only to be used by IsSerializable
	 */
	protected Category ()
	{
		// Does nothing
	}
	
	
	/**
	 * Creates a starting Category, to which keywords can be added
	 * 
	 * @param name			Human readable name of the category, unique at its tree level
	 * @param uniqueName	The Unique name of the Keyword, to use in lookups
	 * @param metadata		The metadata for the keyword, to use as hover text
	 * @param parent		Parent Category, null if this is a top level item
	 */
	public Category (String name, String uniqueName, String metadata, Category parent)
	{
		if (name == null)
		{
			System.err.print ("Bad add, name == null: uniqueName == ");
			System.err.print (uniqueName);
			System.err.print (": metadata == ");
			System.err.print (metadata);
			System.err.print (": parent == ");
			if (parent == null)
				System.err.println (parent);
			else
				System.err.println (parent.toString ());
			name = "";
		}
		
		if (uniqueName == null)
		{
			System.err.print ("Bad add, name == ");
			System.err.print (name);
			System.err.print (": uniqueName == null: metadata = ");
			System.err.print (metadata);
			System.err.print (": parent = ");
			if (parent == null)
				System.err.println (parent);
			else
				System.err.println (parent.toString ());
			uniqueName = "";
		}
		
		this.name = name;
		this.uniqueName = uniqueName;
		this.metadata = metadata;
		this.parent = parent;
		this.keywords = new ArrayList<Keyword> ();
	}
	
	
	/**
	 * @return the human meaningful name
	 */
	public final String getName ()
	{
		return name;
	}
	
	
	/**
	 * @return the uniqueName
	 */
	public final String getUniqueName ()
	{
		return uniqueName;
	}
	
	
	/**
	 * @return the metadata
	 */
	public final String getMetadata ()
	{
		return metadata;
	}
	
	
	/**
	 * @return the parent
	 */
	public final Category getParent ()
	{
		return parent;
	}
	
	
	/**
	 * @return the keywords, guaranteed to never be null, might be empty
	 */
	public final List<Keyword> getKeywords ()
	{
		return keywords;
	}
	
	
	/**
	 * Add a keyword to the Category
	 * 
	 * @param theKeyword	New Keyword.  Will be ignored if null
	 */
	public final void addKeyword (Keyword theKeyword)
	{
		if (theKeyword != null)
			keywords.add (theKeyword);
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString ()
	{
		StringBuilder builder = new StringBuilder ();
		builder.append ("Category [");
		
		if (name != null)
		{
			builder.append ("human name = ");
			builder.append (name);
			builder.append (", ");
		}
		
		if (uniqueName != null)
		{
			builder.append ("uniqueName = ");
			builder.append (uniqueName);
			builder.append (", ");
		}
		
		if (metadata != null)
		{
			builder.append ("metadata = ");
			builder.append (metadata);
			builder.append (", ");
		}
		
		if (parent != null)
		{
			builder.append ("parent = ");
			builder.append (parent);
			builder.append (", ");
		}
		
		if (keywords != null)
		{
			builder.append ("keywords = ");
			builder.append (keywords);
		}
		
		builder.append ("]");
		
		return builder.toString ();
	}
	
}
