/**
 * bior_pipes
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.pipeline;

import static edu.mayo.pipes.JSON.lookup.LookupPipe.kBlankJSON;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.jayway.jsonpath.JsonPath;
import com.tinkerpop.pipes.AbstractPipe;
import edu.mayo.bior.util.BioR;
import edu.mayo.genomicutils.refassembly.*;
import edu.mayo.pipes.JSON.lookup.LookupPipe;
import edu.mayo.pipes.exceptions.InvalidPipeInputException;
import edu.mayo.pipes.history.*;
import edu.mayo.senders.Sender;
import edu.mayo.senders.SystemOutSender;
import java.io.IOException;
import java.util.*;

/**
 * A class that contains a modifiable boolean
 * <p>@author Gregory Dougherty</p>
 *
 */
class ModBool
{
	private boolean	value;
	
	
	/**
	 * @param value
	 */
	ModBool (boolean value)
	{
		this.value = value;
	}
	
	
	/**
	 * @return the value
	 */
	public final boolean isTrue ()
	{
		return value;
	}
	
	
	/**
	 * @param value the value to set
	 */
	public final void setValue (boolean value)
	{
		this.value = value;
	}
}


/**
 * Pipe that takes variable length input, and produces output that can be used by any of the BioR tools.
 * Input can be of the form:</br>
 * 		- rsId</br>
 * 		- rsId, chrom, position</br>
 * 		- rsId, chrom, position, refAllele1, refAllele2</br>
 *
 * <p>@author Gregory Dougherty, 2014-Aug</p>
 */
public class Variant2JSONPipe extends AbstractPipe<History, History>
{
	private LookupPipe		rsIDLookup;
	private RefBasePairLookup	genome;
	protected String			genomeBuild;
	private Sender			sender;
	/** number of the data line being processed (does not include header lines) */
	private int				dataLineNumber = 0;
	private boolean			dumpWarnings = false;
	private int				warningCount = 0;
	private int				errorCount = 0;
	private int				identifierCol = -1;
	private int				allele1Col = -1;
	private int				allele2Col = -1;
	private int				chromosomeCol = -1;
	private int				positionCol = -1;
	/** If have this many cols, might have a chromosome and position */
	private int				allCol = -1;
	private int				posCols = -1;
//	private int				errorCount = 0;
	private int				headerSize = 0;
	private List<History>	mResults = null;
	private boolean			mInitialized = false;
	
	private static String	catalogPath = calcCatalogPath ();
	private static String	genomeBasePath = catalogPath + "/ref_assembly";
	private static Map<String, String[]>	lookupMap = null;
	
	private static final JsonPath	chromPath = JsonPath.compile ("CHROM");
	private static final JsonPath	posPath = JsonPath.compile ("POS");
//	private static final Logger sLogger = Logger.getLogger (Variant2JSONPipe.class);
	
	
	private static final int	kIdentifierCol = 0;
	private static final int	kChromosomeCol = kIdentifierCol + 1;
	private static final int	kPositionCol = kChromosomeCol + 1;
	private static final int	kAllele1Col = kPositionCol + 1;
	private static final int	kAllele2Col = kAllele1Col + 1;
	private static final String	kRsIDStart = "rs";
	private static final int	kRsIDStartLen = kRsIDStart.length ();
	private static final String	kColDelimiter = "\t";
//	private static final String	kPlusStrand = "+";
//	private static final String	kDefaultGenomeBuild = "GRCh37.p13";
	private static final String	kGenomeBuildDefault = "GRCh37";
	private static final String	kDocumentBase = "/research/bsi/data/catalogs/bior/v1";
//	private static final String	kGenomePath = "/research/bsi/data/refdata/ncbi_genome/human/processed/latest/2014_02_04/GRCh37.p13/" + 
//											  "hs_ref_GRCh37.p13.fa.tsv.bgz";
	private static final String	kWarning = "WARNING";
	private static final String	kError = "ERROR";
	private static final int	kWarningLen = kWarning.length ();
	private static final int	kErrorLen = kError.length ();
	private static final String	kNoRSIDFoundWarning = "No match found for ";
	private static final String	kMalformedRSIDError = " is not a valid rsID";
	/** Warning given when rsID given location does not match lookup location */
	public static final String	kPositionMatchFailedWarning = "The rsID found has a different position than was specified";
	/** Warning given when had to flip the alleles */
	public static final String	kFlipAllelesWarning = "Neither of the alleles supplied matched the reference, so swapped strands";
	private static final String[]	kBases = {"A", "C", "G", "T"};
	private static final boolean	kReturnBlanks = false;
	protected static final boolean	kRemoveBlanks = true;
	private static final String[]	kColumnNames = {"ID", "CHROM", "POS", "ALLELE1", "ALLELE2"};
	private static final String[]	kChromosomeLetters = {"X", "Y", "XY", "M", "MT"};
	protected static final int[]	kChromosomeNumbers = {23, 24, 25, 26, 26};
	private static final int		kMinChromosome = 1;
	private static final int		kFirstChangeChromosome = 23;
	private static final int		kNumChangeChromosome = kChromosomeNumbers.length;
	private static final int		kMaxChromosome = 26;
	/** Error message when don't have a header */
	public static final String		kNoHeader = "Must have a tab delimited header as the first non '#' line of input";
	/** Error message when have a header with either a position or a chromosome, but not both */
	public static final String		kBadHeader = "Can not have a header that has only the chromosome or only the position";
	private static final String		kVCFEmpty = ".";
    /** First half of a chromosome error, before the invalid chromosome */
	private static final String	kChrErrorJSON1 = "{\"" + kError + "\":\"Chromosome '";
    /** Last half of a chromosome error, after the invalid chromosome */
	private static final String	kChrErrorJSON2 = "' is not valid\"}";
    /** First half of a position error, before the invalid position */
	private static final String	kPosErrorJSON1 = "{\"" + kError + "\":\"Invalid position '";
    /** Last half of a position error, after the invalid position */
	private static final String	kPosErrorJSON2 = "'\"}";
    /** First half of a chromosome + position error, before the invalid chromosome + position combo */
	private static final String	kChrPosErrorJSON1 = "{\"" + kError + "\":\"Chromosome '";
    /** Last half of a chromosome + position error, in between the invalid chromosome and position combo */
	private static final String	kChrPosErrorJSON2 = "', position '";
    /** Last half of a chromosome + position error, after the invalid chromosome + position combo */
	private static final String	kChrPosErrorJSON3 = "' is not valid\"}";
    /** First half of an allele error, before the invalid allele */
	private static final String	kAlleleErrorJSON1 = "{\"" + kError + "\":\"Invalid allele '";
    /** Last half of an allele error, after the invalid allele */
	private static final String	kAlleleErrorJSON2 = "'\"}";
	private static final String	kInvalidChrPos = ".";
	private static final String[]	kNCBIFile37 = {catalogPath + "/dbSNP/142/00-All_GRCh37.tsv.bgz", 
	                             	               catalogPath + "/dbSNP/142/index/00-All_GRCh37.ID.idx.h2.db"};
	private static final String[]	kNCBIFile38 = {catalogPath + "/dbSNP/142_GRCh38/variants_nodups.v1/00-All.vcf.tsv.bgz", 
													catalogPath + "/dbSNP/142_GRCh38/variants_nodups.v1/index/00-All.ID.idx.h2.db"};
	private static final String[]	kBuilds = {"GRCh37", "GRCh37.p13", "GRCh38"};
	private static final String[][]	kNCBIFiles = {kNCBIFile37, kNCBIFile37, kNCBIFile38};
	/** Index for the lookup file in array returned by {@link #getLookupFiles(String)} */
	public static final int	kLookupFile = 0;
	/** Index for the lookup index file in array returned by {@link #getLookupFiles(String)} */
	public static final int	kLookupIndexFile = kLookupFile + 1;
	
	private JsonParser	jsonParser = new JsonParser ();

	
	/**
	 * Standard Constructor
	 * The {@linkplain LookupPipe} lets us process the rsID lines.</br>
	 * The sender allows Variant2JSONPipe pipe to send errors over a message queue, to stdout, stderr, 
	 * to a file or whatever location the calling class desires.</br>
	 * Defaults to using Genome Build GRCh37
	 * 
	 * @param rsIDLookup		LookupPipe connected to an rsID lookup "database"
	 * @param sender			Where to send the errors, null if should just send to stderr
	 */
	public Variant2JSONPipe (LookupPipe rsIDLookup, Sender sender)
	{
		this (rsIDLookup, sender, kGenomeBuildDefault);
	}
	
	
	/**
	 * Standard Constructor
	 * The {@linkplain LookupPipe} lets us process the rsID lines.
	 * The sender allows Variant2JSONPipe pipe to send errors over a message queue, to stdout, stderr, 
	 * to a file or whatever location the calling class desires.
	 * 
	 * @param rsIDLookup		LookupPipe connected to an rsID lookup "database"
	 * @param sender			Where to send the errors, null if should just send to stderr
	 * @param genomeBuild		Build to use, i.e. "GRCh37.p13"
	 */
	public Variant2JSONPipe (LookupPipe rsIDLookup, Sender sender, String genomeBuild)
	{
		this.rsIDLookup = rsIDLookup;
		
		if (isEmpty (genomeBuild))
			this.genomeBuild = kGenomeBuildDefault;
		else
			this.genomeBuild = genomeBuild;
		
		if (sender != null)
		{
			this.sender = sender;
			dumpWarnings = true;
		}
		else
			this.sender = new SystemOutSender ("stderr"); //by default, send the error to system.err.
	}
	
	
	/**
	 * Get the path to where Variant2JSONPipe will be looking for catalogs
	 * 
	 * @return	The path, guaranteed to not be null
	 */
	public static String getCatalogPath ()
	{
		return catalogPath;
	}
	
	
	/**
	 * Get the paths to the dbSNPs file and index file to use for the specified build
	 * 
	 * @param build	The build to get this time
	 * @return	String[]: 0: Path to the dbSNPs file. 1: Path to index file
	 */
	public static final String[] getLookupFiles (String build)
	{
		if (lookupMap == null)
		{
			lookupMap = new HashMap<String, String[]> ();
			
			int	which = 0;
			
			for (String aBuild : kBuilds)
			{
				lookupMap.put (aBuild, kNCBIFiles[which]);
				++which;
			}
		}
		
		String[]	results = lookupMap.get (build);
		
		if (results == null)
			results = kNCBIFiles[0];
		
		return results;
	}
	
	
	/**
	 * Get the path to where catalogs are located, starting with $BIOR_CATALOG, then going to 
	 * kDocumentBase if $BIOR_CATALOG isn't defined
	 * 
	 * @return	String
	 */
	private static String calcCatalogPath ()
	{
		String	path = System.getenv ("BIOR_CATALOG");
		
		if (path == null)
			BioR.report ("BIOR_CATALOG was null");
		else
			BioR.report ("BIOR_CATALOG was " + path);
		
		if (path != null)
			return path;
		
		return kDocumentBase;
	}
	
	
	/**
	 * Initialize metadata, and clear out any header
	 * 
	 * @return	The next history to process
	 */
	protected History getHistory ()
	{
		if (mInitialized)
			return starts.next ();
		
		History			history = starts.next ();
		List<String>	headerRows = new ArrayList<String> ();
		String			line = history.getMergedData (kColDelimiter);
		
		while (line.startsWith ("#"))
		{
			headerRows.add (line);
			
			if (starts.hasNext ())
			{
				history = starts.next ();
				line = history.getMergedData (kColDelimiter);
			}
			else
			{
				// attempted to grab next line, but there are no more
				// means there are no data lines after header lines
				initializeMetaData (history, headerRows);
				
				// re-throw to tell next pipe there are no more History objects
				break;
			}
		}
		
		// set the metadata
		initializeMetaData (history, headerRows);
		mInitialized = true;
		
		return history;
	}
	
	
	/**
	 * Initializes the History metadata.
	 * 
	 * @param history		History to get initialized
	 * @param headerRows	Rows to add to its metadata
	 */
	private void initializeMetaData (History history, List<String> headerRows)
	{
		HistoryMetaData hMeta = new HistoryMetaData (headerRows);
		
		// process column header if present
		int	size = headerRows.size ();
		if (size > 0)
		{
			String	colHeaderLine = headerRows.get (size - 1);
			
			if (colHeaderLine.startsWith ("#"))
			{
				List<ColumnMetaData>	columns = hMeta.getColumns ();
				
				// trim off leading #
				colHeaderLine = colHeaderLine.substring (1);
				
				for (String colName : colHeaderLine.split (kColDelimiter))
				{
					ColumnMetaData	cmd = new ColumnMetaData (colName);
					columns.add (cmd);
				}
			}
		}
		
		history.setMetaData (hMeta);
	}
	
	
	/* (non-Javadoc)
	 * @see com.tinkerpop.pipes.AbstractPipe#processNextStart()
	 */
	@Override
	protected History processNextStart () throws NoSuchElementException
	{
		if ((mResults != null) && !mResults.isEmpty ())
			return mResults.remove (0);
		
		History history;
		try
		{
			do
			{
				history = starts.next ();
			}
			while (isEmpty (history));
		}
		catch (NoSuchElementException done)
		{
			if (dumpWarnings)
			{
				StringBuilder	message = new StringBuilder ();
				
				message.append ("Had ");
				message.append (warningCount);
				if (errorCount > 0)
				{
					message.append (" warnings and ");
					message.append (errorCount);
					message.append (" errors");
				}
				else
					message.append (" warnings");
				
				sender.write (message.toString ());
			}
			
			throw done;
		}
		
		// record the data line we are going to process
		++dataLineNumber;
		
		if (headerSize == 0)
			getHeaderInformation (history);
		
		// check to make sure we have the required minimum # of columns
		int	numCols = history.size ();
		int	numAdd;	// Pad out history to match the header
		if ((numAdd = (headerSize - numCols)) > 0)
		{
			for (int i = 0; i < numAdd; ++i)
				history.add ("");
		}
		
		if (numCols < allCol)
		{
			if (numCols > posCols)
				return processPosition (history);
			
			if (numCols > identifierCol)
			{
				String	rsID = history.get (identifierCol);
				
				if (isRsID (rsID))
					return processRsID (history, kReturnBlanks);
			}
			
			StringBuilder errorMesg = new StringBuilder ();
			errorMesg.append ("Invalid variant line at data line # ");
			errorMesg.append (dataLineNumber);
			errorMesg.append (".\nThe variant format requires ");
			errorMesg.append (allCol + 1);//kMinNumCols
			errorMesg.append (" fixed fields per data line, but found only ");
			errorMesg.append (numCols);
			errorMesg.append (" field(s).\n");
			errorMesg.append ("Make sure the variant file has the necessary fields delimited by TAB characters.\n");
			errorMesg.append ("Invalid variant line content: \"");
			errorMesg.append (history.getMergedData ("\t"));
			errorMesg.append ("\"");
			
			reportError (errorMesg.toString ());
		}
		
		return processPosition (history);
	}
	
	
	/**
	 * Get the information we need from the MetaData columns
	 * 
	 * @param history	History to get information from, must not be null, must be filled in
	 */
	private void getHeaderInformation (History history)
	{
		HistoryMetaData	metaData = history.getMetaData ();
		
		if (metaData == null)
			reportTerminalError (kNoHeader);
		
		@SuppressWarnings ("null")	// Compiler can't tell reportError () always throws exception
		List<ColumnMetaData>	columns = metaData.getColumns ();
		
		headerSize = columns.size () - 1;	// We add one column to the metadata
		for (int i = 0; i < headerSize; ++i)
		{
			ColumnMetaData	theColumn = columns.get (i);
			String			name = theColumn.getColumnName().toUpperCase ();
			
			if (name.equals (kColumnNames[kChromosomeCol]))
				allCol = Math.max (allCol, chromosomeCol = i);
			else if (name.equals (kColumnNames[kPositionCol]))
				allCol = Math.max (allCol, posCols = Math.max (posCols, positionCol = i));
			else if (name.equals (kColumnNames[kIdentifierCol]))
				allCol = Math.max (allCol, posCols = Math.max (posCols, identifierCol = i));
			else if (name.equals (kColumnNames[kAllele1Col]))
				allCol = Math.max (allCol, allele1Col = i);
			else if (name.equals (kColumnNames[kAllele2Col]))
				allCol = Math.max (allCol, allele2Col = i);
		}
		
		if (allCol < 0)
			reportTerminalError (kNoHeader);
		
		if ((chromosomeCol < 0) ^ (positionCol < 0))
			reportTerminalError (kBadHeader);
	}


	/**
	 * Return true if the String is an rsID, which is to say it has the format rs[integer]
	 * 
	 * @param testStr	String to test
	 * @return	True if it's an rsID, else false
	 */
	private static final boolean isRsID (String testStr)
	{
		if (isEmpty (testStr))
			return false;
		
		if (!testStr.startsWith (kRsIDStart))
			return false;
		
		try
		{
			Long.parseLong (testStr.substring (kRsIDStartLen));
			return true;
		}
		catch (NumberFormatException oops)
		{
			return false;
		}
	}
	
	
	/**
	 * Pass history to rsIDLookup to extract the rsID and look for it in rsIDLookup's DB.  If removeBlanks 
	 * is false, return the results regardless of whether or not anything was found.  Otherwise return 
	 * null if there wasn't a match found, and return the match if it was found
	 * 
	 * @param history		History holding the rsID to look for, that will add the result to
	 * @param removeBlanks	If true, only return matches, if false, return warning if no match
	 * @return	A history with the match found, or null if removeBlanks is true and no match found
	 */
	private History processRsID (History history, boolean removeBlanks)
	{
		// Throw an exception if the header column does not fall between 0 and history.size ()
		if ((identifierCol < 0) || (identifierCol >= history.size ()))
			throw new InvalidPipeInputException ("ERROR: The 'id' header column is missing", this);
		
		mResults = rsIDLookup.processHistory (history, identifierCol);
		
		History	result = mResults.remove (0);
		boolean	wasBlank = removeBlank (result);
		
		if (wasBlank)
		{
			if (removeBlanks)
				return null;
			
			String	rsID = history.get (identifierCol);
			
			if (isRsID (rsID))
				result.add (addWarning (null, kNoRSIDFoundWarning + rsID));
			else
				result.add (addError (null, rsID + kMalformedRSIDError));	// If got here, have a line with just an id, so it's an error
		}
		
		return result;
	}
	
	
	/**
	 * Pass history to rsIDLookup to extract the rsID and look for it in rsIDLookup's DB.  If a match 
	 * isn't found, pass history to positionLookup
	 * 
	 * @param history		History holding the rsID and / or position to look for, that will add the result to
	 * @return	A History with JSON added, or a blank JSON if no match found
	 */
	private History processPosition (History history)
	{
		boolean	hasPosition = (positionCol >= 0) && (positionCol < history.size ());
		boolean	hasIdentifier = (identifierCol >= 0) && (identifierCol < history.size ());
		if (!hasPosition || (hasIdentifier && isRsID (history.get (identifierCol))))
		{
			History	result = processRsID (history, hasPosition);	// kRemoveBlanks if could have a position
			if (result != null)
			{
				if (hasPosition)
					return validateResult (result);
				return result;
			}
		}
		
		mResults = getPossibleHistories (history);
		
		return mResults.remove (0);
	}
	
	
	/**
	 * Validate that the rsID has a chromosome and position that match what the user gave
	 * 
	 * @param result	History holding the information for the rsID, plus the initial user given information
	 * @return	The passed in History.  If there were no problems, it will be unmodified.  If the chromosome 
	 * and / or position do not match, will have a WARNING added to the JSON
	 */
	private History validateResult (History result)
	{
		int		last = result.size () - 1;
		String	json = result.get (last);
		String	chromosome = parseChromosome ((String) chromPath.read (json));
		// NOTE: Due to a change in the dbSNP catalog, the "POS" field has changed from a String to an Integer
		//       "POS" was a String in dbSNP version 142, and an Integer in 150
		//      (use the String.valueOf() method instead of relying on converting from String to String, or Integer to String as in the new case)
		String  position = String.valueOf(posPath.read(json));
		String  chrom = getFromColumn (result, chromosomeCol);
		String  pos = getFromColumn (result, positionCol);
		
		if ((!isVCFEmpty (chrom) && !chrom.equalsIgnoreCase (chromosome)) || (!isVCFEmpty (pos) && !pos.equals (position)))	
			result.set (last, addWarning (json, kPositionMatchFailedWarning));
		
		return result;
	}
	
	
	/**
	 * Get data from a History, returning an empty string if the column < 0
	 * 
	 * @param result	History to get information from.  Must not be null
	 * @param col		Column to get information from.  Will return "" if < 0
	 * @return	A String, possibly empty, never null
	 */
	private String getFromColumn (History result, int col)
	{
		if (col < 0)
			return "";
		
		return result.get (col);
	}
	
	
	/**
	 * Parse a chromosome, returning 1 - 22, X, Y, XY, M for 23 - 26, or M for MT, or else whatever 
	 * was in chromosome
	 * 
	 * @param chromosome	Text to parse
	 * @return	A String, or null if chromosome was null
	 */
	private String parseChromosome (String chromosome)
	{
		if (isVCFEmpty (chromosome))
			return chromosome;
		
		try
		{
			int	chrom = Integer.parseInt (chromosome);
			
			switch (chrom)
			{
				case 23:
					return "X";
					
				case 24:
					return "Y";
					
				case 25:
					return "XY";
					
				case 26:
					return "M";
					
//				default:	// If it's any other number, just return it
//					return chromosome;
			}
		}
		catch (NumberFormatException oops)
		{
			if (chromosome.equalsIgnoreCase ("MT"))
				return "M";
		}
		
		return chromosome;
	}
	
	
	/**
	 * Make a List holding three History objects, one for each possible alt allele that can occur at the 
	 * position specified by history.  The data is added as a JSON that holds all the fields needed by 
	 * other BioR commands, as the last field of the History
	 * 
	 * @param history	Object holding the chromosome and position of interest
	 * @return	List of the history with JSON added for each of the possible alleles
	 */
	private List<History> getPossibleHistories (History history)
	{
		List<History>	results = new ArrayList<History> ();
		String			chrom = getFromColumn (history, chromosomeCol);
		String			pos = getFromColumn (history, positionCol);
		String			allele1 = getFromColumn (history, allele1Col);
		String			allele2 = getFromColumn (history, allele2Col);
		
		if (invalidAllele (allele1) || invalidAllele (allele2))
		{
			StringBuilder	error = new StringBuilder (100);
			
			error.append (kAlleleErrorJSON1);
			if (invalidAllele (allele1))
				error.append (allele1);
			else
				error.append (allele2);
			error.append (kAlleleErrorJSON2);
			results.add (copyAppend (history, error.toString ()));
			sender.write (deJSON (error));
			return results;
		}
		
		if (pos.isEmpty () || !isPositiveInteger (pos))
		{
			StringBuilder	error = new StringBuilder (100);
			
			error.append (kPosErrorJSON1);
			error.append (pos);
			error.append (kPosErrorJSON2);
			results.add (copyAppend (history, error.toString ()));
			sender.write (deJSON (error));
			return results;
		}
		
		ModBool	isValid = new ModBool (false);
		chrom = validChromosome (chrom, isValid);	// Converts 23 - 26 to X, Y, ...
		if (!isValid.isTrue ())
		{
			StringBuilder	error = new StringBuilder (100);
			
			error.append (kChrErrorJSON1);
			error.append (chrom);
			error.append (kChrErrorJSON2);
			results.add (copyAppend (history, error.toString ()));
			sender.write (deJSON (error));
			return results;
		}
		
		try
		{
			if (genome == null)
				genome = getRefAssemblyFinder ();
			
			String	ref = genome.getBasePairAtPosition (chrom, pos, pos);
			if ((ref == null) || ref.equals (kInvalidChrPos))
			{
				StringBuilder	error = new StringBuilder (100);
				
				error.append (kChrPosErrorJSON1);
				error.append (chrom);
				error.append (kChrPosErrorJSON2);
				error.append (pos);
				error.append (kChrPosErrorJSON3);
				results.add (copyAppend (history, error.toString ()));
				sender.write (deJSON (error));
				return results;
			}
			
			int		curBP = Integer.parseInt (pos);
			String	warning = null;
			String	rsID = (identifierCol >= 0) ? history.get (identifierCol) : "";
			
			if (isRsID (rsID))	// Had an rsID, but didn't find a match
				warning = kNoRSIDFoundWarning + rsID;
			
			if (!allele1.isEmpty () || !allele2.isEmpty ())
			{
				addWithAlleles (results, history, chrom, ref, allele1, allele2, curBP, warning);
			}
			else
			{
				for (String alt : kBases)
				{
					if (alt.equals (ref))
						continue;
					
					results.add (copyAppend (history, makeGoldenJSON (chrom, ref, alt, curBP, curBP, warning)));
				}
			}
		}
		catch (NumberFormatException oops)
		{
			oops.printStackTrace ();
			StringBuilder errorMesg = new StringBuilder ();
			
			errorMesg.append ("Invalid variant line at data line # ");
			errorMesg.append (dataLineNumber);
			errorMesg.append (".\nThe variant format requires an integer for position, but got '");
			errorMesg.append (pos);
			errorMesg.append ("'.\n");
			errorMesg.append ("Make sure the variant file has the necessary fields delimited by TAB characters.\n");
			errorMesg.append ("Invalid variant line content: \"");
			errorMesg.append (history.getMergedData ("\t"));
			errorMesg.append ("\"");
			
			reportError (errorMesg.toString ());
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
			reportError ("Could not open Genome Catalog");
		}
		catch (AssemblyNotSupportedException oops)
		{
			oops.printStackTrace ();
			reportError ("Could not open Genome Catalog");
		}
		
		return results;
	}
	
	
	/**
	 * Get the {@linkplain RefBasePairLookup} for the current genomeBuild
	 * 
	 * @return	The correct {@linkplain RefBasePairLookup}
	 * @throws IOException
	 * @throws AssemblyNotSupportedException
	 */
	private RefBasePairLookup getRefAssemblyFinder () throws IOException, AssemblyNotSupportedException
	{
		return genome = getRefAssemblyFinder (genomeBuild);
	}
	
	
	/**
	 * Get the {@linkplain RefBasePairLookup} for the current genomeBuild
	 * 
	 * @param build	The build to get this time
	 * @return	The correct {@linkplain RefBasePairLookup}
	 * @throws IOException
	 * @throws AssemblyNotSupportedException
	 */
	public static final RefBasePairLookup getRefAssemblyFinder (String build) throws IOException, AssemblyNotSupportedException
	{
		// Get the base dir for the ref assembly catalogs from the RefAssemblyFinder class
		String	baseDirRefAssemblyCatalogs = new RefAssemblyFinder ().getBaseDirForRefAssemblyCatalogs ();
		
		// If not found, then use the default
		if (isEmpty (baseDirRefAssemblyCatalogs))
			baseDirRefAssemblyCatalogs = genomeBasePath;
		
		return new RefBasePairLookup (baseDirRefAssemblyCatalogs, build);
	}
	
	
	/**
	 * A valid allele is null, empty, or one of kBases
	 * 
	 * @param theAllele	Allele to test
	 * @return	True if invalid, false if not
	 */
	private boolean invalidAllele (String theAllele)
	{
		if (isEmpty (theAllele))
			return false;
		
		int	size = theAllele.length ();
		if (size == 1)
			return !testAllele (theAllele);
		
		for (int i = 0; i < size; ++i)
		{
			if (!testAllele (theAllele.substring (i, i + 1)))
				return true;
		}
		
		return false;
	}

	
	/**
	 * A valid allele is one of kBases
	 * 
	 * @param theAllele	Allele to test
	 * @return	True if valid, false if not
	 */
	private boolean testAllele (String theAllele)
	{
		for (String testBase : kBases)
		{
			if (testBase.equalsIgnoreCase (theAllele))
				return true;
		}
		
		return false;
	}


	/**
	 * Return true if the String can be parsed as a positive int, false if it can't (includes if have String whose value > maxInt
	 * 
	 * @param test	String to test
	 * @return	True if it's a valid Integer > 0
	 */
	protected static final boolean isPositiveInteger (String test)
	{
		try
		{
			return Integer.parseInt (test) > 0;
		}
		catch (NumberFormatException oops)
		{
			return false;
		}
	}

	
	/**
	 * Return true if the String can be parsed as a non-negative int, false if it can't (includes if have String whose value > maxInt
	 * 
	 * @param test	String to test
	 * @return	True if it's a valid Integer >= 0
	 */
	protected static final boolean isNonNegativeInteger (String test)
	{
		try
		{
			return Integer.parseInt (test) >= 0;
		}
		catch (NumberFormatException oops)
		{
			return false;
		}
	}

	
	/**
	 * Return true if the String can be parsed as an int, false if it can't (includes if have String whose value > maxInt
	 * 
	 * @param test	String to test
	 * @return	True if it's a valid Integer
	 */
	protected static final boolean isInteger (String test)
	{
		try
		{
			Integer.parseInt (test);
			return true;
		}
		catch (NumberFormatException oops)
		{
			return false;
		}
	}


	/**
	 * Take a String in JSON Format, and remove the {} and any ""
	 * 
	 * @param error
	 * @return
	 */
	private static final String deJSON (StringBuilder error)
	{
		int	length;
		if ((error == null) || ((length = error.length ()) == 0))
			return "";
		
		if (length < 2)
			return error.toString ();
		
		if ('{' == error.charAt (0))
		{
			error.delete (0, 1);
			--length;
			
			if ('}' == error.charAt (length - 1))
			{
				error.delete (length - 1, length);
				--length;
			}
		}
		
		int	pos;
		
		while ((pos = error.indexOf ("\"")) >= 0)
			error.delete (pos, pos + 1);
		
		return error.toString ();
	}


	/**
	 * Validate the passed in chromosome.  Valid chromosomes are integers from kMinChromosome to kMaxChromosome, 
	 * or in kChromosomeLetters
	 * 
	 * @param chrom	Chromosome to verify
	 * @return	True if valid, false if not
	 */
	private String validChromosome (String chrom, ModBool isValid)
	{
		try
		{
			int	chr = Integer.valueOf (chrom);
			if ((chr >= kMinChromosome) && (chr <= kMaxChromosome))
			{
				isValid.setValue (true);
				if (chr >= kFirstChangeChromosome)
				{
					for (int i = 0; i < kNumChangeChromosome; ++i)
					{
						if (chr == kChromosomeNumbers[i])
							return kChromosomeLetters[i];
					}
				}
				return chrom;
			}
		}
		catch (NumberFormatException oops)
		{
			String	testChr = chrom.toUpperCase ();
			for (String test : kChromosomeLetters)
			{
				if (test.equals (testChr))
				{
					isValid.setValue (true);
					return chrom;
				}
			}
		}
		
		isValid.setValue (false);
		return chrom;
	}
	
	
	/**
	 * Make one or more Golden JSON based on the passed in alleles and ref allele
	 * 
	 * @param results	History to add the results to, must not be null
	 * @param history	Base History to get columns from, must not be null
	 * @param chrom		Chromosome, must not be null or empty
	 * @param ref		Reference allele, must not be null or empty
	 * @param allele1	One of two alleles to match against
	 * @param allele2	One of two alleles to match against
	 * @param curBP		Location in the chromosome we're using
	 * @param warning	Warning string to add to the JSON, or null if none to add
	 * @throws IOException	If have to re-get ref can get an exception
	 */
	private void addWithAlleles (List<History> results, History history, String chrom, String ref, 
								 String allele1, String allele2, int curBP, String warning) throws IOException
	{
		addWithAlleles (results, history, chrom, ref, allele1, allele2, curBP, warning, true);
	}
	
	
	/**
	 * Make one or more Golden JSON based on the passed in alleles and ref allele
	 * 
	 * @param results	History to add the results to, must not be null
	 * @param history	Base History to get columns from, must not be null
	 * @param chrom		Chromosome, must not be null or empty
	 * @param ref		Reference allele, must not be null or empty
	 * @param allele1	One of two alleles to match against
	 * @param allele2	One of two alleles to match against
	 * @param curBP		Location in the chromosome we're using
	 * @param warning	Warning string to add to the JSON, or null if none to add
	 * @param recurse	If true, can recurse, if false, don't
	 * @throws IOException	If have to re-get ref can get an exception
	 */
	private void addWithAlleles (List<History> results, History history, String chrom, String ref, String allele1, 
								 String allele2, int curBP, String warning, boolean recurse) throws IOException
	{
		String	allele;
		int		endBP = curBP;
		
		// Check for a match, or for length > 1 and 1st char matching, i.e. a deletion
		// An insertion would have the correct allele at allele1, and a string at allele 2
		if (ref.equalsIgnoreCase (allele1))
		{
			allele = allele2;
		}
		else if ((allele1.length () > 1) && ref.equalsIgnoreCase (allele1.substring (0, 1)))	// Deletion
		{
			endBP = curBP + allele1.length () - 1;
			ref = allele1.toUpperCase ();
			allele = allele2;
		}
		else if (ref.equalsIgnoreCase (allele2))
		{
			allele = allele1;
		}
		else
		{
			String	invert1 = invert (allele1);
			if (invert1.equals (allele2))	// A and T or C and G
			{
				results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele1, curBP, endBP, warning)));
				results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele2, curBP, endBP, warning)));
			}
			else if (allele1.equals (allele2) || isEmpty (allele2))	// A and A, A and "", etc..
			{
				results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele1, curBP, endBP, warning)));
			}
			else if (ref.equals ("N"))
			{
				results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele1, curBP, endBP, warning)));
				results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele2, curBP, endBP, warning)));
			}
			else 
			{
				if (recurse)
				{
					if ((allele1.length () > 1) || (allele2.length () > 1))
						addInvertedIndel (results, history, chrom, curBP, allele1, allele2, warning);
					else
						addWithAlleles (results, history, chrom, ref, invert1, invert (allele2), curBP, warning, false);
					
					History	result = results.get (results.size () - 1);
					int		last = result.size () - 1;
					String	json = result.get (last);
					
					result.set (last, addWarning (json, kFlipAllelesWarning));
				}
				else	// Use the initial, not the reversed
				{
					results.add (copyAppend (history, makeGoldenJSON (chrom, ref, invert (allele1), curBP, endBP, warning)));
					results.add (copyAppend (history, makeGoldenJSON (chrom, ref, invert (allele2), curBP, endBP, warning)));
				}
			}
			return;
		}
		
		if (!allele.isEmpty ())
			results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele, curBP, endBP, warning)));
		else
			results.add (copyAppend (history, makeGoldenJSON (chrom, ref, ref, curBP, endBP, warning)));
	}
	
	
	/**
	 * Add an indel to the history when it's been reported from the minus strand
	 * 
	 * @param results	History to add the results to, must not be null
	 * @param history	Base History to get columns from, must not be null
	 * @param chrom		Chromosome, must not be null or empty
	 * @param curBP		Location in the chromosome we're using
	 * @param allele1	One of two alleles to match against, one of these must have length > 1
	 * @param allele2	One of two alleles to match against, one of these must have length > 1
	 * @param recurse	If true, can recurse, if false, don't
	 * @throws IOException	If have to re-get ref can get an exception
	 */
	private void addInvertedIndel (List<History> results, History history, String chrom, int curBP, 
									String allele1, String allele2, String warning) throws IOException
	{
		--curBP;
		String	pos = "" + curBP;
		String	ref = genome.getBasePairAtPosition (chrom, pos, pos);
		String	allele;
		boolean	insert = allele2.length () > 1;
		
		if (insert)
		{
			allele = ref + invert (allele2.substring (1));	// Get everything after first base
			results.add (copyAppend (history, makeGoldenJSON (chrom, ref, allele, curBP, curBP, warning)));
		}
		else
		{
			allele = ref + invert (allele1.substring (1));	// Get everything after first base
			results.add (copyAppend (history, makeGoldenJSON (chrom, allele, ref, curBP, curBP, warning)));
		}
		
	}
	
	
	/**
	 * Return the complimentary allele.  If a string of alleles, return the reverse compliment
	 * 
	 * @param allele	The allele(s) to complement, must not be null or empty
	 * @return	The complement, or "C"
	 */
	private static final String invert (String allele)
	{
		int	size = allele.length ();
		
		if (size == 1)
			return invertBase (allele);
		
		StringBuilder	inverted = new StringBuilder (size);
		
		for (int i = size; i > 0; --i)
			inverted.append (invertBase (allele.substring (i - 1, i)));
		
		return inverted.toString ();
	}
	
	
	/**
	 * Return the complimentary allele
	 * 
	 * @param allele	The allele to complement, must not be null
	 * @return	The complement, or "C"
	 */
	private static final String invertBase (String allele)
	{
		if (allele.equals ("A"))
			return "T";
		
		if (allele.equals ("T"))
			return "A";
		
		if (allele.equals ("C"))
			return "G";
		
		return "C";
	}
	
	
	/**
	 * Make a JSON with all the fields needed by a BioR tool, including the alt allele
	 * 
	 * @param history	Object holding the chromosome and position of interest
	 * @param warning	Warning string to add to the JSON, or null if none to add
	 * @return	String with the Chromosome, position, and Ref allele, looking up the plus strand 
	 * Ref allele for the passed in chromosome and position
	 */
	private static final String makeGoldenJSON (String chrom, String ref, String alt, int start, int end, String warning)
	{
		StringBuilder	result = new StringBuilder (30);
		
		result.append ('{');
		
		addJSONString (result, "_landmark", chrom, false, false);
		addJSONString (result, "_refAllele", ref, true, false);
		addJSONString (result, "_altAlleles", alt, true, true);
		addJSONInteger (result, "_minBP", start, true);
		addJSONInteger (result, "_maxBP", end, true);
		if (warning != null)
			addJSONString (result, kWarning, warning, true, false);
		
		result.append ('}');
		
		return result.toString ();
	}
	
	
	/**
	 * Report a warning to sender
	 * 
	 * @param json		String to add to.  If not null or empty, must be valid JSON with {} at ends
	 * @param warning	Warning message to return.  Must not be null, should not be empty
	 * @return	A properly constructed JSON with a warning as its last element
	 */
	private String addWarning (String json, String warning) throws InvalidPipeInputException
	{
		++warningCount;
		if (isJSONEmpty (json))
		{
			StringBuilder	result = new StringBuilder (warning.length () + kWarningLen + 6);
			
			result.append ("{\"");
			result.append (kWarning);
			result.append ("\":\"");
			result.append (warning);
			result.append ("\"}");
			
			return result.toString ();
		}
		
		JsonElement	root = this.jsonParser.parse (json);
		
		root.getAsJsonObject ().addProperty (kWarning, warning);
		return root.toString ();
	}
	
	
	/**
	 * Report an error to sender in the JSON
	 * 
	 * @param json	String to add to.  If not null or empty, must be valid JSON with {} at ends
	 * @param error	Error message to return.  Must not be null
	 */
	private String addError (String json, String error) throws InvalidPipeInputException
	{
		// First, print the error to stderr
		System.err.print (kError);
		System.err.print (':');
		System.err.println (error);
		
		++errorCount;
		if (isJSONEmpty (json))
		{
			StringBuilder	result = new StringBuilder (error.length () + kErrorLen + 6);
			
			result.append ("{\"");
			result.append (kError);
			result.append ("\":\"");
			result.append (error);
			result.append ("\"}");
			
			return result.toString ();
		}
		
		JsonElement	root = this.jsonParser.parse (json);
		
		root.getAsJsonObject ().addProperty (kError, error);
		return root.toString ();
	}
	
	
	/**
	 * Report an error to sender, then throw an InvalidPipeInputException with the error
	 * 
	 * @param error	Error message to return.  Must not be null, should not be empty
	 * @throws InvalidPipeInputException	The exception we always throw
	 */
	private void reportError (String error) throws InvalidPipeInputException
	{
		sender.write (error);
		throw new InvalidPipeInputException (error, this);
	}
	
	
	/**
	 * Report a non-recoverable error to sender, then throw an InvalidPipeInputException with the error
	 * 
	 * @param error	Error message to return.  Must not be null, should not be empty
	 * @throws InvalidPipeInputException	The exception we always throw
	 */
	private void reportTerminalError (String error) throws InvalidPipeInputException
	{
//		sender.write (error);
		throw new RuntimeException (error);
	}
	
	
	/**
	 * Add a JSON String key and value to StringBuilder
	 * 
	 * @param result		StringBuilder to add to, can not be null
	 * @param key			key value (i.e. "_landmark"). Must not be null, should not be empty
	 * @param value			value for the key (i.e. "1"). Must not be null
	 * @param havePrevious	If true, will put a comma before the new key:value pair
	 * @param multi			If true, then the field is one that might have multiple values, so will 
	 * put [] around the value
	 */
	private static final void addJSONString (StringBuilder result, String key, String value, boolean havePrevious, boolean multi)
	{
		if (havePrevious)
			result.append (',');
		
		result.append ('"');
		result.append (key);
		result.append ("\":");
		
		if (multi)
			result.append ('[');
		
		result.append ('"');
		result.append (value);
		result.append ('"');
		
		if (multi)
			result.append (']');
	}
	
	
	/**
	 * Add a JSON integer key and value to StringBuilder
	 * 
	 * @param result		StringBuilder to add to, can not be null
	 * @param key			key value (i.e. "_landmark"). Must not be null, should not be empty
	 * @param value			value for the key (i.e. 1)
	 * @param havePrevious	If true, will put a comma before the new key:value pair
	 */
	private static final void addJSONInteger (StringBuilder result, String key, int value, boolean havePrevious)
	{
		if (havePrevious)
			result.append (',');
		
		result.append ('"');
		result.append (key);
		result.append ("\":");
		result.append (value);
	}
	
	
	/**
	 * Test a History to see if the last thing added to it was a blank JSON.  If it was, remove it
	 * 
	 * @param history	History to test.  Must not be null
	 * @return	True if it ended with a blank that was removed, false if it didn't not end with a blank
	 */
    private static final boolean removeBlank (History history)
    {
    	int		size = history.size () - 1;
    	String	last = history.get (size);
    	
    	if (kBlankJSON.equals (last))
    	{
    		history.remove (size);
        	return true;
    	}
    	
    	return false;
    }
    
    
    /**
     * Make a copy of this History and append the passed in string to the end of it
     * 
     * @param history	History to add to, must not be null
     * @param result	Result to add, should not be null
     * @return	The copied and added History
     */
    private static final History copyAppend (History history, String result)
	{
		History clone = (History) history.clone ();
		clone.add (result);
		return clone;
	}
    
    
 	/**
 	 * Utility function to test if a string is null or empty
 	 * 
 	 * @param testStr	String to test
 	 * @return	True if string is null or empty, else false
 	 */
 	private static final boolean isEmpty (String testStr)
 	{
 		return (testStr == null) || testStr.isEmpty ();
 	}
 	
	
	/**
	 * Test if we got a {@linkplain History} object from a blank line
	 * 
	 * @param history	History object to test
	 * @return	True is null, size 0, or size 1 with an empty elements
	 */
 	private static final boolean isEmpty (History history)
	{
		if (history == null)
			return true;
		
		int	size = history.size ();
		if (size == 0)
			return true;
		
		if (size > 1)
			return false;
		
		return isEmpty (history.get (0));
	}
	
    
 	/**
 	 * Utility function to test if a string is null or empty or only has a "."
 	 * 
 	 * @param testStr	String to test
 	 * @return	True if string is null or empty or only has a ".", else false
 	 */
 	private static final boolean isVCFEmpty (String testStr)
 	{
 		return (testStr == null) || testStr.isEmpty () || testStr.equals (kVCFEmpty);
 	}
 	
    
 	/**
 	 * Utility function to test if a JSON string is null or empty.  Any string lacking ':' will be considered empty
 	 * 
 	 * @param testStr	String to test
 	 * @return	True if string is null or empty, else false
 	 */
 	private static final boolean isJSONEmpty (String testStr)
 	{
 		if (testStr == null)
 			return true;
 		
 		return (testStr.indexOf (':') < 0);
 	}
 	
}
