package edu.mayo.bior.pipeline;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/** Data for the ##INFO metadata lines at the top of a VCF file.
 *  This is used by bior_tjson_to_vcf to construct the ##INFO lines that are created when adding columns to the INFO column
 * 	@author Michael Meiners (m054457)
 */
public class InfoMetaObject {
	//enum KeyValType { String, Number, Boolean };
	enum KeyValType { Flag, Integer, Float, Character, String };

	/** The header column key plus any nested keys within JSON within a column.
	 *  If the column is not JSON, the key will be the same as parentColumnName (ex: "CHROM").
	 *  If the column is JSON, it will be the column header name plus the nested JSON key
	 *  (ex: say column header is "bior.dbsnp" and JSON key in data coumn is "INFO.SSR", then key: "bior.dbsnp.INFO.SSR", parentColumnName: "bior.dbsnp")  */
	public String jsonKey;
	public List<String> val = new ArrayList<String>();
	public String parentColumnName;
	public KeyValType type;
	/** "." = 1 or more (variable),  0 = boolean flag,  1 = exactly one every time,  2 = exactly two every time, etc... */
	public String numOccurrences = ".";
	public String description = "";
	
	public InfoMetaObject(String aJsonKey, List<String> aVals, String aParentColumn, KeyValType aType, String numOccurrences, String description) {
		this.jsonKey = aJsonKey;
		this.val = aVals;
		this.parentColumnName = aParentColumn;
		this.type = aType;
		this.numOccurrences = numOccurrences;
		this.description = description;
	}
	
	public InfoMetaObject() { }
	
	/** Construct a ##INFO line from the given info, looking up description, type, and other info from the BioR header map.
	 *  Examples:  
	 *  	##INFO=<ID=CAVA_ALTCLASS,Number=.,Type=String,Description="Alternate CLASS annotation">
	 *  	##INFO=<ID=MQ,Number=1,Type=Float,Description="RMS Mapping Quality">
	 */
	public String toString() {
		return String.format("##INFO=<ID=%s,Number=%s,Type=%s,Description=\"%s\">", 
				getId(),
				this.numOccurrences,
				this.type.toString(),
				this.description );
	}

	/** Create the ID from the parentColumnName and the key.  If the column did not contain JSON, then the ID is just the parentColumnName 
	 * @return  Id generated from a combination of jsonKey and parentColumnName  */
	public String getId() {
		return getId(this.jsonKey, this.parentColumnName);
	}
	
	/** Create the ID from a given parentColumnName and jsonKey 
	 * @param aJsonKey  The JSON key (can be single or nested, such as INFO or INFO.AF.EUR) used within a JSON column
	 * @param aParentColumnName The header for the column
	 * @return  Id generated from a combination of jsonKey and parentColumnName  */
	public static String getId(String aJsonKey, String aParentColumnName) {
		// Parent column name should always be filled in:
		String id = "";
		if(aParentColumnName != null && aParentColumnName.length() > 0)
			id = aParentColumnName;
		
		// If column contained JSON, it will also have a key
		if( aJsonKey != null && aJsonKey.length() > 0 ) {
			if( id.length() > 0 )
				id += ".";
			id += aJsonKey;
		}
		
		return id;		
	}

	

}
