package edu.mayo.bior.cli.cmd;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.log4j.lf5.LogLevel;

import edu.mayo.bior.catalog.verification.CatalogVerifier;
import edu.mayo.bior.catalog.verification.CatalogVerifier.Phase;
import edu.mayo.bior.catalog.verification.CatalogVerifier.VAL_TYPE;
import edu.mayo.bior.catalog.verification.MessageLogger;
import edu.mayo.bior.catalog.verification.VerifierExecutionException;
import edu.mayo.bior.catalog.verification.VerifierInputException;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.cli.InvalidOptionArgValueException;

public class VerifyCatalogCommand implements CommandPlugin
{

   protected static final String OPTION_CATALOG_FILE = "d";
   protected static final String OPTION_OUTPUT_FILE  = "o";
   protected static final String OPTION_SAMPLE_SIZE  = "s";
   // Ex:  Phase.METADATA, Phase.ORDER, Phase.JSON
   protected static final String OPTION_PHASE  	     = "phase";
   protected static final String OPTION_START_LINE   = "startLine";
   protected static final String OPTION_NUM_LINES    = "numLines";

   private String cmd;
   
   public void init(Properties props) throws Exception
   {
		cmd = props.getProperty("command.name");
   }

   public void execute(CommandLine cmdLine, Options opts) throws Exception
   {
      String catalogFilePath = cmdLine.getOptionValue(OPTION_CATALOG_FILE);
      
      verifyCatalogExists(opts, catalogFilePath);

      String outputFilePath = getAndVerifyOutputPath(cmdLine, opts);
      MessageLogger messageLogger = new MessageLogger((outputFilePath == null)  ?  getDefaultLogFilePath(catalogFilePath)  :  outputFilePath);

      long sampleSize = getSampleSize(cmdLine);

      List<CatalogVerifier.Phase> phasesToExecute = getPhasesToExecute(cmdLine);
      
      // We may skip the first X lines, and start part way into the file.
      // This is not applicable to the METADATA phase as that verifies files outside the catalog
      // It will generally not be used by the ORDER phase as that is fairly fast running.
      // The StartLine and NumLines flags should be used together
      long startLine = getStartLine(cmdLine);
      long numLines  = getNumLinesToProcess(cmdLine);
      
      
      CatalogVerifier catalogValidator = new CatalogVerifier();
      try {
         catalogValidator.verify(catalogFilePath, VAL_TYPE.STRICT, messageLogger, sampleSize, phasesToExecute, startLine, numLines);
      } catch (VerifierInputException inputE) {
    	  logAndThrowException(inputE, messageLogger, "Input exception raised while verifying catalog: " + catalogFilePath + " Exception: " + inputE.getMessage(), 5);
      } catch (VerifierExecutionException execE) {
    	  logAndThrowException(execE, messageLogger, "Execution exception raised while verifying catalog: " + catalogFilePath + " Exception: " + execE.getMessage(), 6);
      } catch (Exception e) {
    	  logAndThrowException(e, messageLogger, "General error while verifying catalog: " + catalogFilePath + " Exception: " + e.getMessage(), 7);
      }
   }


   private String getDefaultLogFilePath(String catalogFilePath) {
	   String currentWorkingDir = System.getProperty("user.dir");
	   String filename = new File(catalogFilePath).getName().replace(".tsv.bgz", "_verify.txt");
	   File verifyOutFile = new File(currentWorkingDir, filename);
	   return verifyOutFile.getAbsolutePath();
   }

   private void logAndThrowException(Exception inputE, MessageLogger messageLogger, String errorMsg, int errorCode)
		throws Exception {
       System.err.println(errorMsg);
       if( messageLogger != null )
    	   messageLogger.logError(errorMsg, errorCode);
       inputE.printStackTrace();
       throw inputE;
   }

   private long getSampleSize(CommandLine line) {
	   if( line.hasOption(OPTION_SAMPLE_SIZE) ) {
		   long oneInXLines = Long.parseLong(line.getOptionValue(OPTION_SAMPLE_SIZE));
		   
		   if( oneInXLines < 1 ) {
			   throw new IllegalArgumentException("The sample size must be at least 1");
		   }
		   
		   if( oneInXLines > 1 ) {
			   System.err.println("Warning: Only 1 in " + oneInXLines + " will be sampled, so some issues may be missed.");
		   }
		   return oneInXLines;
	   }
	   // Option not provided so return default=1 (sample every line)
	   return 1L;
   }
   
   private List<Phase> getPhasesToExecute(CommandLine line) {
	   if( line.hasOption(OPTION_PHASE) ) {
		   String phaseStrCsv = line.getOptionValue(OPTION_PHASE);
		   return getPhaseListFromCsv(phaseStrCsv);
	   } else {
		   return CatalogVerifier.getAllPhases();
	   }
   }
   
   

   /** Given a comma-separated list of Phase names, return them as a list
    *  (ex: "METADATA,ORDER,JSON" to [Phase.METADATA, Phase.ORDER, Phase.JSON];
    *  or  "METADATA" to [Phase.METADATA] ) */
   private List<Phase> getPhaseListFromCsv(String phaseStrCsv) {
	   List<Phase> phaseList = new ArrayList<Phase>();
	   for(String phase : phaseStrCsv.split(",")) {
		   String phaseTrimmed = phase.trim();
		   if( phaseTrimmed.length() > 0 ) {
			   if( isValidPhase(phaseTrimmed) )
				   phaseList.add(Phase.valueOf(phaseTrimmed));
			   else
				   throw new IllegalArgumentException("No valid arguments were supplied for the -" + OPTION_PHASE + " flag: " + phase);
		   }
	   }
	   if( phaseList.size() == 0 )
		   throw new IllegalArgumentException("No valid arguments were supplied for the -" + OPTION_PHASE + " flag: " + phaseStrCsv);
	   return phaseList;
   }

   private boolean isValidPhase(String phaseTrimmed) {
	   for(Phase p : Phase.values()) {
		   if( p.name().equals(phaseTrimmed) )
			   return true;
	   }
	   return false;
   }

   /** Return the command line option value as a String, or NULL if not defined */
   private String getOptionAsString(CommandLine cmdLine, String optionFlag) {
	   if( cmdLine.hasOption(optionFlag) )
		   return cmdLine.getOptionValue(optionFlag);
	   return null;
   }
   
   /* 1-based start line for where to jump to within the catalog file to begin verification */
   private long getStartLine(CommandLine cmdLine) {
	   return getStartLine(getOptionAsString(cmdLine, OPTION_START_LINE));
   }
   
   /* 1-based start line for where to jump to within the catalog file to begin verification */
   protected long getStartLine(String startLineStr) {
	   // If not specified, use 1
	   if( startLineStr == null )
		   return 1;

	   long startLine = 1;
	   try {
		   startLine = Long.parseLong(startLineStr);
	   } catch(Exception e) {
		   throw new IllegalArgumentException("Bad argument for " + cmd + " --" + OPTION_START_LINE + ": " + startLineStr);
	   }

	   if( startLine <= 0 )
		   throw new IllegalArgumentException(cmd + " argument for --" + OPTION_START_LINE + " must be a positive number.  Was: " + startLineStr);
	   
	   return startLine;
   }

   /* Number of lines to verify (after the starting line) */
   private long getNumLinesToProcess(CommandLine cmdLine) {
	   return getNumLinesToProcess(getOptionAsString(cmdLine, OPTION_NUM_LINES));
   }
	   
   /* Number of lines to verify (after the starting line) */
   protected long getNumLinesToProcess(String numLinesStr) {
	   // If not specified, use max Long
	   if( numLinesStr == null )
		   return Long.MAX_VALUE;

	   long numLines = Long.MAX_VALUE;
	   try {
		   numLines = Long.parseLong(numLinesStr);
	   } catch(Exception e) {
		   throw new IllegalArgumentException("Bad argument for verify's NumLines -" + OPTION_NUM_LINES + ": " + numLinesStr);
	   }
	   
	   if( numLines < 0 )
		   throw new IllegalArgumentException(cmd + " argument for --" + OPTION_START_LINE + " must be 0 (for all lines), or a positive number.  Was: " + numLines);

	   if( numLines == 0 )
		   numLines = Long.MAX_VALUE;
	   
	   return numLines;
   }


   private String getAndVerifyOutputPath(CommandLine line, Options opts)
		throws InvalidOptionArgValueException, IOException
   {
	   String outputFilePath = null;
	   if (line.hasOption(OPTION_OUTPUT_FILE)) {
		   outputFilePath = line.getOptionValue(OPTION_OUTPUT_FILE);
		   File outputFile = new File(outputFilePath);
		   File parentDir = outputFile.getParentFile();
		   if (parentDir != null) {
			   if (!parentDir.exists()) {
				   String dirPath = parentDir.getPath();
	               String msg = String.format("The directory '%s' for the output file does not exist.", dirPath);
	               throw new InvalidOptionArgValueException(opts.getOption(OPTION_OUTPUT_FILE), outputFilePath, msg);
			   }
			   if (!parentDir.canWrite()) {
				   String dirPath = parentDir.getPath();
	               String msg = String.format("The directory '%s' for the output file is not writable.", dirPath);
	               throw new InvalidOptionArgValueException(opts.getOption(OPTION_OUTPUT_FILE), outputFilePath, msg);
	           }
		   }
	   }
	   return outputFilePath;
   }

   private void verifyCatalogExists(Options opts, String catalogFilePath) throws InvalidOptionArgValueException {
	if (!new File(catalogFilePath).exists())
      {
         String msg = "The catalog file path '" + catalogFilePath + "' does not exist. Please specify a valid catalog file path.";
         throw new InvalidOptionArgValueException(opts.getOption(OPTION_CATALOG_FILE), catalogFilePath, msg);
      }
}
}
