package edu.mayo.bior.cli.cmd;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.log4j.Logger;

import com.tinkerpop.pipes.Pipe;
import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.pipeline.UnixStreamPipeline;
import edu.mayo.bior.pipeline.Variant2JSONPipe;
import edu.mayo.bior.pipes.history.MakeFirstLineHeaderPipe;
import edu.mayo.pipes.util.BiorProperties;
import edu.mayo.pipes.util.BiorProperties.Key;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.cli.InvalidOptionArgValueException;
import edu.mayo.pipes.JSON.lookup.LookupPipe;
import edu.mayo.pipes.JSON.lookup.lookupUtils.IndexUtils;
import edu.mayo.pipes.UNIX.CatPipe;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryInPipe;
import edu.mayo.pipes.history.HistoryOutPipe;
import edu.mayo.pipes.util.metadata.Metadata;

/**
 * Class to create a tool that will take a file of "variants" and built TJson that can be used by the 
 * rest of BioR.<br/>
 * Variants have the format:<br/>
 * Column 1: Identifier. Either an rsID, or else a name we can safely ignore<br/>
 * Column 2: Chromosome. If Column 1 has an rsID, this is ignored.  If not, the Chromosome of interest<br/>
 * Column 3: Position. If Column 1 has an rsID, this is ignored.  If not, the position on the chromosome of interest<br/>
 * <br/>
 * If the variant is a chromosome and position, will generate one line for each possible ALT allele
 * 
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 */
public class Variant2JSONCommand implements CommandPlugin
{
	private static Logger sLogger = Logger.getLogger (Variant2JSONCommand.class);

	
	private UnixStreamPipeline	mPipeline = new UnixStreamPipeline ();
	private String				operation;
	
	private static final String	kOptionInputFile = "f";
	private static final String	OPTION_INDEX_FILE = "i";
	private static final String	OPTION_CATALOG_FILE = "d";
	// JSON path to extract key (if column is specified, the json in that column is used.
	// If not, then ID is the key)
	private static final String	OPTION_KEY = "p";
	private static final String	kOptionBuild = "b";
	private static final String	kDefaultKey = "ID";
	private static final String	kDefaultCatalog = "/dbSNP/142/00-All_GRCh37.tsv.bgz";
	
	
	/* (non-Javadoc)
	 * @see edu.mayo.cli.CommandPlugin#init(java.util.Properties)
	 */
	public void init (Properties props) throws Exception
	{
		operation = props.getProperty ("command.name");
	}
	
	
	/* (non-Javadoc)
	 * @see edu.mayo.cli.CommandPlugin#execute(org.apache.commons.cli.CommandLine, org.apache.commons.cli.Options)
	 */
	public void execute (CommandLine line, Options opts) throws Exception
	{
		try
		{
			String catalogFilePath = getCatalogFile (line);
			if (!new File (catalogFilePath).exists ())
			{
				throw new InvalidOptionArgValueException (opts.getOption (OPTION_CATALOG_FILE), catalogFilePath,
														  "The catalog file path '" + catalogFilePath + 
														  "' does not exist. Please specify a valid catalog file path.");
			}
			
			String key = kDefaultKey;
			if (line.hasOption (OPTION_KEY))
				key = line.getOptionValue (OPTION_KEY);
			
			String build = null;
			if (line.hasOption (kOptionBuild))
				build = line.getOptionValue (kOptionBuild);
			
			String indexFilePath = "";
			File	indexFile = null;
			boolean isIndexSpecified = line.hasOption (OPTION_INDEX_FILE);
			if (isIndexSpecified)
			{
				indexFilePath = line.getOptionValue (OPTION_INDEX_FILE);
			}
			else
			{
				// find the index file based on catalog-name
				try
				{
                    indexFilePath = IndexUtils.getH2DbIndexPath(catalogFilePath, key);
                    indexFile = new File(indexFilePath);
                    if (!indexFile.exists()) {  // if current, correct index name not found, look for previously named index to see if that exists.
                           indexFilePath = IndexUtils.getH2DbIndexPath_backwardCompatabilityOnly(catalogFilePath, key);
                           indexFile = new File(indexFilePath);
                    }
				}
				catch (IOException ioe)
				{
					throw new InvalidOptionArgValueException (opts.getOption (OPTION_CATALOG_FILE), indexFilePath, 
															  "Error locating the index file");
				}
			}
			
			String	defaultIndexNotExistMsg = "The built-in index for " + key + " does not exist.  " + 
											  "Please e-mail the BioR development team at bior@mayo.edu to add this, " + 
											  "or create your own custom index using the bior_index command.";
			String	specifiedIndexNotExistMsg = "The index file path you specified does not exist: " + indexFilePath;
			String	specifiedIndexNotReadableMsg = "The index file path you specified does not have read access: " + indexFilePath + 
												   ".  Please verify file permissions.";
			if (!indexFile.exists ())
			{
				throw new InvalidOptionArgValueException (opts.getOption (OPTION_INDEX_FILE), indexFilePath, 
						isIndexSpecified ? specifiedIndexNotExistMsg : defaultIndexNotExistMsg, isIndexSpecified);
			}
			else if (!indexFile.canRead ())
			{
				throw new InvalidOptionArgValueException (opts.getOption (OPTION_INDEX_FILE), indexFilePath, 
														  specifiedIndexNotReadableMsg, isIndexSpecified);
			}
			Metadata	metadata = new Metadata (operation);
			String		inputPath = line.getOptionValue (kOptionInputFile);
			String		inputNotExistMsg = "The input file path you specified does not exist: " + inputPath;
			String		inputNotReadMsg = "The input file path you specified does not have read access: " + inputPath + 
										  ".  Please verify file permissions.";
			File		inputFile = (inputPath == null) ? null : new File (inputPath);
			if ((inputFile != null) && !inputFile.exists ())
				throw new InvalidOptionArgValueException (opts.getOption (kOptionInputFile), indexFilePath, inputNotExistMsg, true);
			else if ((inputFile != null) && !indexFile.canRead ())
				throw new InvalidOptionArgValueException (opts.getOption (kOptionInputFile), indexFilePath, inputNotReadMsg, true);
			
			// For this to have a correct description of what we're doing
//			metadata.setCmdType (CmdType.ToTJson);
			Pipe<String, String>	addHeader = new MakeFirstLineHeaderPipe ();
			Pipe<String, History>	setup = new HistoryInPipe (metadata);
			Pipe<String, History>	preLogic = new Pipeline<String, History> (addHeader, setup);
				
//			LookupPipe				rsISLookup = new LookupPipe (lookupFile, lookupIndexFile);
			LookupPipe				rsISLookup = new LookupPipe (catalogFilePath, indexFilePath);
			Pipe<History, History>	logic = new Variant2JSONPipe (rsISLookup, null, build);
			Pipe<History, String>	postLogic = new HistoryOutPipe ();
			
			if (inputPath != null)
				mPipeline.execute (preLogic, logic, postLogic, new CatPipe (), Arrays.asList (inputPath));
			else
				mPipeline.execute (preLogic, logic, postLogic);
		}
		catch (Exception e)
		{
			System.err.println ("ERROR: " + e.getMessage ());
			throw e;
		}
	}


	/**
	 * 
	 * @param line
	 * @return
	 * @throws IOException 
	 */
	private String getCatalogFile (CommandLine line) throws IOException
	{
		if (line.hasOption (OPTION_CATALOG_FILE))
			return line.getOptionValue (OPTION_CATALOG_FILE);
		
		String	catalogDir = System.getenv ("BIOR_CATALOG");
		if (catalogDir == null) {
			logEnvironmentVariables();
			sLogger.warn("BIOR_CATALOG not passed in, and not found in the system environment variables.  Pulling it from bior.properties...");
			catalogDir = getCatalogDirFromBiorProps();
		}
		
		String catalogPath = catalogDir + kDefaultCatalog;
		sLogger.warn("BIOR_CATALOG = " + catalogPath);
		
		return catalogPath;
	}


	private String getCatalogDirFromBiorProps() throws IOException {
		return new BiorProperties().get(Key.fileBase);
	}


	private void logEnvironmentVariables() {
		Map<String, String>	env = System.getenv ();
		sLogger.info("Environment variables:");
		for (Entry<String, String> entry : env.entrySet ()) {
			sLogger.info(entry.getKey () + ":= " + entry.getValue ());
		}		
	}
	
	
	
}
