package edu.mayo.bior.cli.cmd;

import edu.mayo.bior.buildcatalog.BuildInfoKey;
import edu.mayo.bior.catalog.CatalogDataSource;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.markdown.MarkdownInfo;
import edu.mayo.bior.catalog.markdown.transformer.comparison.ComparisonStatsImpl;
import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaDataOperations;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.*;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.escapeChrs;
import static edu.mayo.bior.cli.cmd.CommandUtil.FileAttributes;
import static edu.mayo.bior.cli.cmd.CommandUtil.FileAttributes.EXISTS;
import static edu.mayo.bior.cli.cmd.CommandUtil.FileAttributes.READABLE;
import static edu.mayo.bior.cli.cmd.CommandUtil.handleFile;
import static java.util.Arrays.asList;
import static org.apache.commons.io.FileUtils.readFileToString;


/** Generates markdown documentation about a catalog using the datasource.properties, columns.tsv, and potentially a previous markdown file */
public class CatalogMarkdownCommand implements CommandPlugin {

    private static final String OPTION_CATALOG_BGZ_FILE = "d";
    private static final String OPTION_PREVIOUS_MARKDOWN_FILE = "p";
    private static final String OPTION_OUTPUT_DIR = "o";
	private static final String OPTION_COMPARISON_JSON_FILE = "c";


    // Put markdown in the current directory
    private File defaultOutputDir = new File(".");

    enum MarkdownTemplateKeys { CATALOG_NAME, DEPRECATION_NOTICE, DATASOURCE_PROPERTIES, COLUMNSTSV_PROPERTIES, FAQ_PROPERTIES, CATALOG_PATH, REFDATA_PATH};
    
    
    @Override
    public void init(Properties properties) throws Exception {
    }

    
    @Override
    public void execute(CommandLine cl, Options options) throws Exception {
    	File catalogFile = getCatalogFile(cl);
        
        File outputDir = getOutputDir(cl);
        
        File previousMarkdown = getPreviousMarkdown(cl);

        File comparisonJsonFile = getComparisonJsonFile(cl);
        
        MarkdownInfo markdownStr = createMarkdown(catalogFile, outputDir, previousMarkdown, comparisonJsonFile);
    }

    private File getCatalogFile(CommandLine cl) throws Exception {
        List<FileAttributes> catalogFileAttrs = asList(EXISTS, FileAttributes.READABLE);
        File catalogFile = handleFile(cl, OPTION_CATALOG_BGZ_FILE, catalogFileAttrs);
        return catalogFile;
	}


	private File getPreviousMarkdown(CommandLine cl) {
    	File outputDir = cl.hasOption(OPTION_PREVIOUS_MARKDOWN_FILE) ?  new File(cl.getOptionValue(OPTION_PREVIOUS_MARKDOWN_FILE)) : defaultOutputDir;
    	if (!outputDir.exists()) {
    		outputDir.mkdirs();
    	}
    	return outputDir;
	}


	private File getOutputDir(CommandLine cl) throws Exception {
    	File outputDir = cl.hasOption(OPTION_OUTPUT_DIR) ?  new File(cl.getOptionValue(OPTION_OUTPUT_DIR)) : defaultOutputDir;
    	if (!outputDir.exists()) {
    		outputDir.mkdirs();
    	}
    	// Verify that directory exists now and is executable, readable, and writeable
    	if( ! (outputDir.canRead()  &&  outputDir.canExecute()  &&  outputDir.canWrite()) ) {
    		throw new RuntimeException("Markdown output directory must be readable, writeable, and executable");
    	}
    	return outputDir;
	}

	/**
	 * Attempts to get the comparison JSON file.
	 * @param cl Apache commons {@link CommandLine}
	 * @return Returns {@link File} pointing to the JSON file.  NULL if the option was not specified on the CLI.
	 */
	private File getComparisonJsonFile(CommandLine cl) throws Exception {
		if (cl.hasOption(OPTION_COMPARISON_JSON_FILE)) {
			return handleFile(cl, OPTION_COMPARISON_JSON_FILE, asList(EXISTS, READABLE));
		} else {
			return null;
		}
	}

	/** Create the markdown file in the output directory.  Generally the output directory should be a "markdown" subdirectory within the target directory */
	public MarkdownInfo createMarkdown(File catalogFile, File outputDir, File previousMarkdownFile, File comparisonJsonFile) throws IOException, CatalogFormatException, TransformException {
		String catalogMarkdownTemplate = loadCatalogMarkdownTemplate();
		
		MarkdownInfo markdownInfo = loadMarkdownInfo(catalogFile);
		
		// If the catalog has been deprecated, then add deprecation notes to the template
		markdownInfo.markdownOutputStr = catalogMarkdownTemplate
				.replace(MarkdownTemplateKeys.CATALOG_NAME.toString(), 			escapeChrs(markdownInfo.datasourceProperties.getShortUniqueName()))
				.replace(MarkdownTemplateKeys.DEPRECATION_NOTICE.toString(), 	markdownInfo.deprecationNotes)
				.replace(MarkdownTemplateKeys.DATASOURCE_PROPERTIES.toString(), getDatasourcePropertiesMarkdown(markdownInfo))
				.replace(MarkdownTemplateKeys.COLUMNSTSV_PROPERTIES.toString(), getColumnsTsvMarkdown(markdownInfo))
				.replace(MarkdownTemplateKeys.CATALOG_PATH.toString(), 			escapeChrs(catalogFile.getCanonicalPath()))
				.replace(MarkdownTemplateKeys.REFDATA_PATH.toString(), 			isExistsAndReadable(markdownInfo.refdataSourceDir)  ? escapeChrs(markdownInfo.refdataSourceDir.getCanonicalPath()) : "")
				;

		List<MarkdownTransformer> transformers = new ArrayList<MarkdownTransformer>();
		transformers.add(new ComparisonStatsImpl(comparisonJsonFile));

		// apply transformations to markdown
		for (MarkdownTransformer transformer: transformers) {
			markdownInfo.markdownOutputStr = transformer.transform(markdownInfo.markdownOutputStr);
		}

		// Create the target directory
		outputDir.mkdirs();
		
		// Write the markdown string to the output directory
		markdownInfo.markdownOutputFile = new File(outputDir, catalogFile.getName().replace(".tsv.bgz", ".md"));
		FileUtils.write(markdownInfo.markdownOutputFile, markdownInfo.markdownOutputStr);
		
		return markdownInfo;
    }

	private String loadCatalogMarkdownTemplate() throws IOException {
		String path = this.getClass().getResource("/catalogMarkdownTemplate.md").getPath();
		return readFileToString(new File(path));
	}


	private MarkdownInfo loadMarkdownInfo(File catalogFile) throws CatalogFormatException, IOException {
		MarkdownInfo markdownInfo = new MarkdownInfo();
		
		markdownInfo.catalogTsvBgzFile = catalogFile;
		
		markdownInfo.refdataSourceDir = getRefDataSourceFromBuildInfoTxt(catalogFile);
		
		// Load the <CATALOG>.datasource.properties file
		File datasourcePropsPath = new File(catalogFile.getParentFile(), catalogFile.getName().replace(".tsv.bgz", ".datasource.properties")); 
		markdownInfo.datasourceProperties = new CatalogDataSource(datasourcePropsPath);
		
		// Load the <CATALOG>.columns.tsv file
		File columnsTsv = new File(catalogFile.getParentFile(), catalogFile.getName().replace(".tsv.bgz", ".columns.tsv")); 
		markdownInfo.columns = new ColumnMetaDataOperations(columnsTsv).loadAsList();
				
		// Load deprecation notes if any.  These can be put into a "code-block" section within the markdown text
		markdownInfo.deprecationNotes = loadDeprecationNotes(catalogFile);
		
		return markdownInfo;
	}

	/** Can possibly get refdata directory from the build/build_info.txt file if it's there, 
	    AND it contains the key MAKE_JSON_ARGS
	    AND it has a value
	    Then get the first argument
	    Else return null 
	 * @throws IOException */
	private File getRefDataSourceFromBuildInfoTxt(File catalogFile) throws IOException {
		File buildInfoTxtFile = new File(catalogFile.getParentFile().getCanonicalPath() + "/build", "build_info.txt");
		
		if( ! isExistsAndReadable(buildInfoTxtFile) ) {
			return null;
		}
		
		// Else, load the build_info.txt properties, and pull out the MAKE_JSON_ARGS value, split by spaces, and get the first value
		Properties datasrcProps = new Properties();
		FileInputStream fin = new FileInputStream(buildInfoTxtFile);
		datasrcProps.load(fin);
		if( fin != null ) {
			fin.close();
		}
		String makeJsonArgs = datasrcProps.getProperty(BuildInfoKey.MAKE_JSON_ARGS.toString()).trim();
		String[] args = makeJsonArgs.split(" ");
		
		if( args.length == 0  ||  args[0].trim().length() == 0 ) {
			return null;
		}
		File refDataDir = new File(args[0].trim());
		if( ! isExistsAndReadable(refDataDir) ) {
			return null;
		}
		return refDataDir;
	}

	protected String getDatasourcePropertiesMarkdown(MarkdownInfo markdownInfo) {
		StringBuilder str = new StringBuilder("Property | Values\n");
		str.append("------------ | ------------\n");
		str.append("ShortUniqueName" 		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getShortUniqueName()) 	+ "\n");
		str.append("Description"     		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getDescription()) 		+ "\n");
		str.append("Source"         		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getSource()) 			+ "\n");
		str.append("Dataset"        		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getDataset()) 			+ "\n");
		str.append("Version"         		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getVersion()) 			+ "\n");
		str.append("Build"           		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getBuild()) 				+ "\n");
		str.append("Format"          		+ " | " + escapeChrs(markdownInfo.datasourceProperties.getFormat()) 			+ "\n");
		str.append("Human Build/Assembly" 	+ " | " + escapeChrs(markdownInfo.datasourceProperties.getHumanBuildAssembly().toString()) + "\n");
		return str.toString();
	}
	
	protected String getColumnsTsvMarkdown(MarkdownInfo markdownInfo) {
		StringBuilder str = new StringBuilder("ColumnName | Type | Count | Description | HumanReadableName\n");
		str.append("------------ | ------------ | ------------ | ------------ | ------------\n");
		for(ColumnMetaData colMeta : markdownInfo.columns) {
			str.append(escapeChrs(colMeta.getColumnName()) 		+ " | "
					+  escapeChrs(colMeta.getType().toString())	+ " | "
					+  escapeChrs(colMeta.getCount()) 			+ " | "
					+  escapeChrs(colMeta.getDescription()) 	+ " | "
					+  escapeChrs(colMeta.getHumanReadableName()) +  "\n");
		}
		return str.toString();
	}

	/** Load the catalog deprecation notes if there are any.  If no deprecation file, then return ""
		Example path:  dbSNP/142_GRCh37.p13/variants_nodups.v1/00-All.vcf.DEPRECATED.txt
		Content is free-text, so may want to isolate to a markdown code-block section 
	 * @throws IOException */ 
	private String loadDeprecationNotes(File catalogFile) throws IOException {
		File deprecationFile = new File(catalogFile.getParentFile(), catalogFile.getName().replace(".tsv.bgz", ".DEPRECATED.txt"));
		
		if( ! isExistsAndReadable(deprecationFile) ) {
			return "";
		}
		
		// For each line in the notes, prefix it with "> " to blockquote the notes
		String notes = "> " + escapeChrs(readFileToString(deprecationFile)).replaceAll("\n", "\n\n> ");
		
		// Deprecation warning message.  The link portion provides hovertext describing deprecation
		String warningMsg = "**[DEPRECATED](link \"A catalog may be deprecated if it contains errors, inconsistencies, or ambiguous fields that are fixed in a subsequent catalog\")\n"
					+		"This catalog is deprecated, and should no longer be used. Please see the latest version of this catalog.**\n\n";

		return warningMsg + notes;
	}
	
	private boolean isExistsAndReadable(File f) {
		return  f != null  &&  f.exists()  &&  f.canRead();
	}
}
