package edu.mayo.bior.catalog.verification;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Map.Entry;

import com.google.gson.*;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.CatalogTabixEntry;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.log4j.lf5.LogLevel;

import htsjdk.tribble.readers.TabixReader;

public class VerifyUtils
{
   private static final Logger sLogger = Logger.getLogger(VerifyUtils.class);

   public static final String NL = System.getProperty("line.separator");
   public static final String TAB = "\t";
   
   private static JsonParser jp = new JsonParser();
   
   protected static String composeDateTime()
   {
      return new SimpleDateFormat("h:mm:ss a 'on' MMM d, yyyy").format(Calendar.getInstance().getTime());
   }


   /**
    * readCatalogRow: Navigates the complexities of reading in multiple types of
    * catalogs. Also populates a CatalogTabixEntry object correctly in these
    * different situations.
    * <p>
    * Non-positional catalogs supported:
    * These will have no tabix column entries (columns 1-3) or designated values
    * that mean they are empty. I need to document exactly what is supported here
    * in the code and by BIOR. I'm not 100% sure the code is supporting what it needs
    * to yet.
    * <p>
    * UNKNOWN	0	0    		<<< As of 4/19/2016, this format is not supported b/c doesn't return the
    * values when using the tabix command.
    * UNKNOWN	0	1    		<<< As of 4/19/2016, this format IS supported b/c it does return the
    * values when using the tabix command.
    * .	0	0    				<<< As of 4/19/2016, this format is not supported b/c doesn't return the
    * values when using the tabix command.
    *
    * @param lineInCatalog
    * @return CatalogTabixEntry populated with the elements of the catalog line, or null
    * if unsupported catalog line format.
    */
   protected static CatalogTabixEntry readCatalogRow(String lineInCatalog, MessageLogger logger)
   {
      if (lineInCatalog == null)
      {
         return null;
      }

      CatalogTabixEntry tabixEntry = null;
      try
      {
         tabixEntry = new CatalogTabixEntry(lineInCatalog);
      }
      catch (CatalogFormatException e)
      {
         logger.logError(e.getMessage(), VerifyErrorCodes.BAD_CATALOG_FORMAT);
      }
      return tabixEntry;
   }

   static JsonObject getJsonObject(String jsonAsString) throws JsonParseException
   {
      JsonObject catalogRowJsonObj = null;
      try
      {
         catalogRowJsonObj = jp.parse(jsonAsString).getAsJsonObject();
      }
      // I think there are a variety of exceptions that can get thrown from here so I'm just going to try and rethrow
      // it as a JsonParseException to see how that works
      catch (Exception e)
      {
         throw new JsonParseException(String.format("Trouble parsing '%s'", jsonAsString), e);
      }
      return catalogRowJsonObj;
   }

   // TODO - note that parentJsonKey is never used so that should be resolved somehow
   protected static Object getJsonValueForJsonKey(com.google.gson.JsonObject jsonObj, String keyOfValueToRetrieve,
                                                  String parentJsonKey)
   {
      java.lang.Object currentJsonObj = null;
      for (Entry<String, JsonElement> thisEntry : jsonObj.entrySet())
      {
         String jsonKey = thisEntry.getKey();
         // TODO - this doesn't seem right to contain the keyOfValueToRetrieve as this could make problems
         if (!(jsonKey.equals(keyOfValueToRetrieve) || jsonKey.contains(keyOfValueToRetrieve)))
         {
            //System.err.println("Not the key [" + jsonKey + "] we are looking for: " + keyOfValueToRetrieve + NL);
            continue;
         }
         else
         {
            JsonElement jsonElem = thisEntry.getValue();
            if (jsonElem.isJsonNull())
            {
               System.err.println("getJavaObjectForJsonID(): Value for json key [" + jsonKey + "] is empty.");
            }
            else if (jsonElem.isJsonObject())
            {
               // Embedded JSON:
               com.google.gson.JsonObject embeddedJsonObj = jsonElem.getAsJsonObject();
               currentJsonObj = getJsonValueForJsonKey(embeddedJsonObj, keyOfValueToRetrieve, jsonKey);
               continue; // If it was a nested json object, we've checked it against the column info already so move on to next column.
            }
            else if (jsonElem.isJsonArray())
            {
               JsonArray jsonArrVal = jsonElem.getAsJsonArray();
               currentJsonObj = jsonArrVal;  // TODO what if these are nested json objects??? Are we doing enough here?
            }
            else if (jsonElem.isJsonPrimitive())
            {
               JsonPrimitive jsonPrimVal = jsonElem.getAsJsonPrimitive();
               if (jsonPrimVal.isBoolean())
               {
                  Boolean boolVal = jsonPrimVal.getAsBoolean();
                  currentJsonObj = boolVal;
               }
               else if (jsonPrimVal.isNumber())
               {
                  currentJsonObj = jsonNumberToJavaNumber(jsonPrimVal);
               }
               else if (jsonPrimVal.isString())
               {
                  String strVal = jsonPrimVal.getAsString();
                  if (strVal != null)
                  {
                     currentJsonObj = strVal;
                  }
               }
            } // end else it is a primitive

            break;  // if we found our key, see if we successfully got an object (value ) for that key
         } // end else we found the jsonkey we are looking for
      }  // for loop to loop through available json

      return currentJsonObj;
   }

   protected static boolean isTabixRetrievalSuccessful(TabixReader tabixRdr,
                                                       CatalogTabixEntry tabixCatEntry, boolean firstResult)
      throws IOException, CatalogFormatException
   {
      boolean successfulMatch = false;   // assume failure until we find an exact match
      boolean allResultsRelevant = true; // assume success unless we see a specific failure

      TabixReader.Iterator tabixItr = getTabixIterator(tabixRdr, tabixCatEntry);
      if (tabixItr == null)
      {
         successfulMatch = false;
         allResultsRelevant = false;
      }
      else
      {
         int currentIteratorIdx = 0;
         int iteratorIdxThatMatched = -1;
         long numTabixResults = 0;

         String tabixLine = null;
         while ((tabixLine = tabixItr.next()) != null)
         {
            currentIteratorIdx++;
            numTabixResults++;

            if (tabixLine.equals(tabixCatEntry.getLine()))
            {
               successfulMatch = true;
               iteratorIdxThatMatched = currentIteratorIdx;
            }
            else
            {
               CatalogTabixEntry qryResultTabixEntry = new CatalogTabixEntry(tabixLine);
               boolean tabixResultMinPosInRange = withinRange(qryResultTabixEntry.getMinPosition(), tabixCatEntry.getMinPosition(), tabixCatEntry.getMaxPosition());
               boolean tabixResultMaxPosInRange = withinRange(qryResultTabixEntry.getMaxPosition(), tabixCatEntry.getMinPosition(), tabixCatEntry.getMaxPosition());
               if (tabixResultMinPosInRange || tabixResultMaxPosInRange)
               {
                  // this is ok
               }
               else if (qryResultTabixEntry.getMinPosition() <= tabixCatEntry.getMinPosition() &&
                  qryResultTabixEntry.getMaxPosition() >= tabixCatEntry.getMaxPosition())
               {
                  // this is ok if a query result is larger than the catalog minBP-maxBP range we are validating.
               }
               else
               {
                  // this is a problem
                  // TODO - put this to the log4j log?
                  String msg = "Tabix query result catalog entry not within minBP or maxBP query range: " + NL +
                     TAB + " queried minBP: " + tabixCatEntry.getMinPosition() + NL +
                     TAB + " queried maxBP: " + tabixCatEntry.getMaxPosition() + NL +
                     TAB + " result minBP out of range: " + qryResultTabixEntry.getMinPosition() + NL +
                     TAB + " result maxBP out of range: " + qryResultTabixEntry.getMaxPosition() + NL;

                  allResultsRelevant = false;
               }
            }
         } // end while processing tabix result rows

         if (successfulMatch && firstResult && iteratorIdxThatMatched == 1)
         {
            // we are ok: we matched a result and the result matched was the first result in iterator; keep successfulMatch set to true.
         }
         else if (successfulMatch && !firstResult && iteratorIdxThatMatched == numTabixResults)
         {
            // we are ok: we matched a result and the result matched was the last result in iterator; keep successfulMatch set to true.
         }
         else
         {
            successfulMatch = false;   // the result that matched is false, or was not in the expected position of the results returned (first or last, depending)
         }
      } // end if found tabix results for query

      if (successfulMatch && allResultsRelevant)
      {
         return true;
      }
      else
      {
         return false;
      }
   }

   /**
    * isEmpty:  checks whether input string is null or empty
    *
    * @param str
    * @return returns true if input string is null or has a trimmed length equal to zero.
    */
   public static boolean isEmpty(String str)
   {
      return StringUtils.isBlank(str);
   }

   /**
    * Does the string represent the mitochondrial chromosome?
    * @param str incoming string representation of chromosome
    * @return true if string is case-insensitive "M" or "MT" after removing leading and trailing space and removing leading
    *    "chr" (case-insensitive). Return false if str is null
    */
   public static boolean isChrM(String str)
   {
      if (str == null)
      {
         return false;
      }
      String strippedStr = str.trim();
      String replacedStr = strippedStr.replaceFirst("^(?i)chr", "").toUpperCase();
      return replacedStr.equals("M") || replacedStr.equals("MT");
   }

   protected static TabixReader.Iterator getTabixIterator(TabixReader tabixRdr, CatalogTabixEntry tabixRowEntry)
   {
      return getTabixIterator(tabixRdr, tabixRowEntry.getChromosome(), tabixRowEntry.getMinPosition(), tabixRowEntry.getMaxPosition());
   }

   protected static TabixReader.Iterator getTabixIterator(TabixReader tabixRdr, String chr, Long start, Long stop)
   {
      TabixReader.Iterator chrItr;
      try
      {
         chrItr = tabixRdr.query(chr + ":" + start + "-" + stop);
      }
      catch (Throwable t)
      {
         sLogger.warn("Exception while requesting tabix iterator for region [" + chr + ":" + start + "-" + stop + "]." +
            " Exception is: " + t.getMessage());
         return null;
      }

      return chrItr;
   }

   protected static Object jsonNumberToJavaNumber(com.google.gson.JsonPrimitive jsonNumber)
   {

      java.lang.Object correctInstanceOfJavaNumber = null;

      java.lang.Number num = stringToJavaNumber(jsonNumber.getAsString());
      if (num instanceof Long)
      {
         correctInstanceOfJavaNumber = (Long) num;
      }
      else if (num instanceof Integer)
      {
         correctInstanceOfJavaNumber = (Integer) num;
      }
      else if (num instanceof BigDecimal)
      {
         correctInstanceOfJavaNumber = (BigDecimal) num;
      }
      else if (num instanceof Double)
      {
         correctInstanceOfJavaNumber = (Double) num;
      }
      return correctInstanceOfJavaNumber;
   }

   /* TODO: Code review and unit test this.
      We should maybe do following order:
               1. See if it fits into an int
    */
   protected static java.lang.Number stringToJavaNumber(String value)
   {
      if (!value.contains("."))
      {
         try
         {
            long longValue = Long.parseLong(value);
            if (longValue >= Integer.MIN_VALUE && longValue <= Integer.MAX_VALUE)
            {
               return (int) longValue;  // if value fits into an int return it as an int
            }
            else
            {
               return longValue;
            }
         }
         catch (NumberFormatException ignored)
         {
            // Ignore exceptions so that it will try to create a double below.
         }
      }

      try
      {
         Double valueAsADouble = Double.parseDouble(value);
         Float valueAsFloat = Float.parseFloat(value);
         if (valueAsADouble >= Float.MIN_VALUE && valueAsADouble <= Float.MAX_VALUE)
         {
            return valueAsFloat;
         }
         else
         {
            return valueAsADouble;
         }
      }
      catch (NumberFormatException nf)
      {
         return new BigDecimal(value);
      }
   }

   /**
    * trim: Verify that the String passed is not null and has
    * a trimmed length greater than zero.
    *
    * @param inputStr input string to trim
    * @return Returns trimmed string. Returns null if inputStr was null.
    */
   public static String trim(String inputStr)
   {
      if (inputStr == null)
      {
         return null;
      }
      return inputStr.trim();
   }

   private static boolean withinRange(long queryPosition, long minRange, long maxRange)
   {
      return queryPosition >= minRange && queryPosition <= maxRange;
   }

}
