package edu.mayo.bior.catalog.stats;

import edu.mayo.pipes.history.ColumnMetaData;

import java.io.PrintWriter;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.mockito.internal.matchers.CompareTo;

public class StatsPrinter {

    public static void printStats(PrintWriter w, CatalogStats catalogStats, String column, ColumnMetaData metaData) {
        printMetaDataHeader(w, metaData);
        w.println();
        printColumnCharacterStats(w, catalogStats, column);
        w.println();
        printColumnValueStats(w, catalogStats, column);
    }

    private static NumberFormat getPercentFormat() {
        NumberFormat defaultFormat = NumberFormat.getPercentInstance();
        defaultFormat.setMinimumFractionDigits(1);
        return defaultFormat;
    }

    private static void printColumnValueStats(PrintWriter w, CatalogStats catalogStats, String column) {

        w.println("# VALUE STATS (sorted by descending frequency)");
        w.println();

        NumberFormat defaultFormat = getPercentFormat();

        CatalogColumnStats colStats = catalogStats.getColumnStats(column);

        // sort by frequency descending, tiebreaker is sorting by value alphabetically
        Comparator<ValueSampling> frequencyComparator = new Comparator<ValueSampling>() {
            @Override
            public int compare(ValueSampling o1, ValueSampling o2) {
                int comparison = ((Long)o2.getFrequency()).compareTo(o1.getFrequency());
                if (comparison == 0) {
                    // tiebreaker is sorting alphabetically
                    return o1.getValue().compareTo(o2.getValue());
                } else {
                    return comparison;
                }
            }
        };
        List<ValueSampling> valueSamplings = new ArrayList<ValueSampling>();
        valueSamplings.addAll(colStats.getValueSamplings());
        Collections.sort(valueSamplings, frequencyComparator);

        final String format = "\t%1$-12s %2$-8s %3$s";

        w.println(String.format(format, "VALUE_COUNT", "VALUE_%", "VALUE"));
        for (ValueSampling valueSampling: valueSamplings) {
            String valCount   = valueSampling.getFrequency() + "";
            String valPercent = defaultFormat.format((double)valueSampling.getFrequency()/(double)colStats.getNumEntries());
            w.println(String.format(format, valCount, valPercent, valueSampling.getValue()));
        }
    }

    private static void printColumnCharacterStats(PrintWriter w, CatalogStats catalogStats, String column) {

        w.println("# CHARACTER STATS");
        w.println();

        NumberFormat defaultFormat = getPercentFormat();

        CatalogColumnStats colStats = catalogStats.getColumnStats(column);

        if (catalogStats.getStoppedReadingAfterChunk())
        {
            w.println(String.format("\tTotal Lines read:         %s (stopped reading after chunk due to large catalog size)", catalogStats.getTotalDataLineCount()));
        }
        else
        {
            w.println(String.format("\tTotal Lines in file:      %s", catalogStats.getTotalDataLineCount()));
        }
        w.println(String.format("\tNum lines sampled:        %s", catalogStats.getNumLinesSampled()));
        w.println(String.format("\tLines that had column:    %s (%s)", colStats.getNumEntries(), defaultFormat.format((double)colStats.getNumEntries()/(double)catalogStats.getNumLinesSampled())));
        w.println(String.format("\tTotal column value chars: %s", colStats.getNumCharacters()));
        w.println("");

        final String format = "\t%1$15s %2$15s %3$15s %4$15s %5$15s";

        List<String> alphanumeric = new ArrayList<String>();
        List<String> nonalphanumeric = new ArrayList<String>();
        long[] asciiStatsTotal = colStats.getTotalAsciiStats().counts;
        long[] asciiStatsLine  = colStats.getLineAsciiStats().counts;
        for (int i=0; i < asciiStatsTotal.length; i++) {
            char c = (char) i;
            long totalCount = asciiStatsTotal[i];
            if (totalCount > 0) {
                long lineCount = asciiStatsLine[i];
                double totalCountPercent = ((double)totalCount / (double)colStats.getNumCharacters());
                double lineCountPercent = ((double)lineCount / (double)colStats.getNumEntries());
                String s = String.format(format,
                        getDisplayValue(c),
                        lineCount,
                        defaultFormat.format(lineCountPercent),
                        totalCount,
                        defaultFormat.format(totalCountPercent)
                );
                if (Character.isLetterOrDigit(c))
                    alphanumeric.add(s);
                else
                    nonalphanumeric.add(s);
            }
        }
        w.println("\tNON-Alphanumeric");
        w.println(String.format(format, "CHARACTER", "LINE_COUNT", "LINE_%", "CHAR_COUNT", "CHAR_%"));
        for (String s: nonalphanumeric) w.println(s);

        w.println("");
        w.println("\tAlphanumeric");
        w.println(String.format(format, "CHARACTER", "LINE_COUNT", "LINE_%", "CHAR_COUNT", "CHAR_%"));
        for (String s: alphanumeric) w.println(s);
    }

    private static String getDisplayValue(char c) {
        if (Character.isISOControl(c)) {
            return getControlCharacterDesc(c);
        }
        else if (Character.isWhitespace(c)) {
            return "<space>";
        } else {
            return String.valueOf(c);
        }
    }

    private static String getControlCharacterDesc(char controlChar) {
        switch ((int)controlChar) {
            case 0:
                return "null";
            case 1:
                return "start of header";
            case 2:
                return "start of text";
            case 3:
                return "end of text";
            case 4:
                return "end of transmission";
            case 5:
                return "enquiry";
            case 6:
                return "acknowledge";
            case 7:
                return "bell";
            case 8:
                return "backspace";
            case 9:
                return "horizontal tab";
            case 10:
                return "line feed";
            case 11:
                return "vertical tab";
            case 12:
                return "form feed";
            case 13:
                return "enter / carriage return";
            case 14:
                return "shift out";
            case 15:
                return "shift in";
            case 16:
                return "data link escape";
            case 17:
                return "device control 1";
            case 18:
                return "device control 2";
            case 19:
                return "device control 3";
            case 20:
                return "device control 4";
            case 21:
                return "negative acknowledge";
            case 22:
                return "synchronize";
            case 23:
                return "end of trans. block";
            case 24:
                return "cancel";
            case 25:
                return "end of medium";
            case 26:
                return "substitute";
            case 27:
                return "escape";
            case 28:
                return "file separator";
            case 29:
                return "group separator";
            case 30:
                return "record separator";
            case 31:
                return "unit separator";
            case 127:
                return "delete";
            default:
                return "UNKNOWN control character ASCII code " + (int) controlChar;
        }
    }

    private static void printMetaDataHeader(PrintWriter w, ColumnMetaData md) {
        w.println("# COLUMN METADATA");
        w.println();

        final String format = "\t%1$-12s %2$s";
        w.println(String.format(format, "COLUMN",      md.getColumnName()));
        w.println(String.format(format, "TYPE",        md.getType()));
        w.println(String.format(format, "COUNT",       md.getCount()));
        w.println(String.format(format, "DESCRIPTION", md.getDescription()));
    }
}
