package edu.mayo.bior.catalog.markdown.transformer.comparison;

import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;

import java.io.PrintWriter;
import java.io.StringWriter;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.chompTrailingNewlines;
import static edu.mayo.bior.catalog.markdown.MarkdownUtils.positiveNumPrint;
import static edu.mayo.bior.catalog.markdown.transformer.comparison.ComparisonStatsImpl.NOT_AVAILABLE_MESG;
import static java.lang.String.format;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Builds a markdown table about catalog volume.
 */
public class CatalogVolumeImpl implements MarkdownTransformer {

    static final String TEMPLATE_KEYWORD = "COMPARISON_STATISTICS_VOLUME_TABLE";

    private CatalogDiffStatistics stats;

    CatalogVolumeImpl(CatalogDiffStatistics stats) {
        this.stats = stats;
    }

    @Override
    public String transform(String markdown) throws TransformException {
        if (stats == null) {
            return markdown.replace(TEMPLATE_KEYWORD, NOT_AVAILABLE_MESG);
        } else {

            return markdown.replace(TEMPLATE_KEYWORD, chompTrailingNewlines(buildTable()));
        }
    }

    private String buildTable() {

        long netDifference = stats.getSummaryStats().getCatalogNewEntries() - stats.getSummaryStats().getCatalogOldEntries();

        StringWriter sWtr = new StringWriter();
        PrintWriter pWtr = new PrintWriter(sWtr);

        pWtr.println("Metric | Description | Count");
        pWtr.println("--- | --- | ---");
        pWtr.println( format("%s | %s | %d",
                        "Previous Rows",
                        "# of rows in the previous catalog",
                        stats.getSummaryStats().getCatalogOldEntries()
                ) );
        pWtr.println( format("%s | %s | %s",
                        "Removed Rows",
                        "# of rows removed from the previous catalog",
                        positiveNumPrint(-1 * stats.getSummaryStats().getCatalogEntriesRemoved())
                ) );
        pWtr.println( format("%s | %s | %s",
                        "Added Rows",
                        "# of rows added to the previous catalog",
                        positiveNumPrint(stats.getSummaryStats().getCatalogEntriesAdded())
                ) );
        pWtr.println( format("%s | %s | %d",
                        "Current Rows",
                        "# of rows in the current catalog",
                        stats.getSummaryStats().getCatalogNewEntries()
                ) );
        pWtr.println( format("%s | %s | %s",
                        "Net Difference",
                        "Difference between Previous Rows and Current Rows",
                        positiveNumPrint(netDifference)
                ) );
        closeQuietly(pWtr);
        closeQuietly(sWtr);
        return sWtr.toString();
    }
}
