package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;

/** A Logger for each step in the Catalog build process
 * @author Michael Meiners (m054457)
 * 2016-04-27  */
public class StepLogger {
	private static Logger sLogger = Logger.getLogger(StepLogger.class);
	
	
	private File mLogFile;

	private File mSummaryFile;
	
	private PrintWriter mLogWriter;
	private PrintWriter mSummaryWriter;
	
	protected static final String DATE_FORMAT_STR = "yyyy-MM-dd HH:mm:ss  ";
	private SimpleDateFormat mDateFormat = new SimpleDateFormat(DATE_FORMAT_STR);
	
	/** The "[BuildCatalog.BUILD_SUBDIR]/progress/" sub directory under the working directory where the catalog will be built */
	public static final String PROGRESS_SUBDIR = BuildCatalog.BUILD_SUBDIR + "/progress";

	
	
	/** Constructor - assumes the log file will be written to [CurrentWorkingDirectory]/[BuildCatalog.BUILD_SUBDIR]/progress/
	 * @param stepFilePrefix  This determines the prefix for the log file that will be written to.  
	 * 			NOTE: This should be the string equivalent of one of the enum values in BuildCatalog.BuildStepKey  
	 * 			Ex:  This is the "CREATE_CATALOG" log in the path: [CurrentWorkingDirectory]/[BuildCatalog.BUILD_SUBDIR]/progress/CREATE_CATALOG.log
	 * @throws IOException */
	public StepLogger(String stepFilePrefix) throws IOException {
		this(stepFilePrefix, new File(".").getCanonicalFile());
	}

	/** Constructor - 
	 * @param stepFilePrefix  This determines the log file that will be written to.  Ex:  [<code>catalogOutputDirectory</code>]/[BuildCatalog.BUILD_SUBDIR]/progress/[stepFilePrefix].log 
	 * @param catalogOutputDirectory The catalog output directory.  The logs will be written to <code>catalogOutputDirectory</code>/[BuildCatalog.BUILD_SUBDIR]/progress/[stepFilePrefix].log  for instance. 
	 * @throws IOException */
	public StepLogger(String stepFilePrefix, File catalogOutputDirectory) throws IOException {
		File logDir = new File(catalogOutputDirectory, PROGRESS_SUBDIR);
		logDir.mkdirs();
		
		mLogFile	 = new File(logDir, stepFilePrefix + ".log");
		mSummaryFile = new File(logDir, stepFilePrefix + ".summary");
		
		mLogWriter 		= new PrintWriter(mLogFile);
		mSummaryWriter 	= new PrintWriter(mSummaryFile);
	}

	/** Log a message to the current step log (as well as the log4j log file).  These messages will NOT be displayed to the user.
	 *  Ex: write to [step].log
	 * @param msg  Message to write to the log	 */
	public void log(String msg) {
		mLogWriter.println(mDateFormat.format(new Date()) + msg);
		mLogWriter.flush();
		sLogger.info(msg);
	}
	
	/** Write a line to the summary file.  Any text written here will be displayed to the user.
	 *  Ex: write to [step].summary
	 * @param msg  Message to write to the summary file	 */
	public void summary(String msg) {
		System.out.println(msg);
		mSummaryWriter.println(msg);
		mSummaryWriter.flush();
	}
	
	/** Log a message and write it to the summary file
	 *  Ex: write to [step].log  and [step].summary
	 * @param msg  Message to write to the log and summary files	 */
	public void logAndSummary(String msg) {
		log(msg);
		summary(msg);
	}
	
	
	/** Read the summary log file and return its contents.  This will return an empty string if the summary file does not yet exist (has not been written to yet)
	 * @return  The contents of the summary file
	 * @throws IOException 
	 */
	public String getSummary() throws IOException {
		if( mSummaryFile.exists() )
			return FileUtils.readFileToString(mSummaryFile);
		else 
			return "";
	}

	/** Get the log file 
	 * @return  The log file */
	public File getLogFile() {
		return mLogFile;
	}

	/** Get the summary file
	 * @return  The summary file */
	public File getSummaryFile() {
		return mSummaryFile;
	}

	/** Close the log and summary file-writer pointers */
	public void closeLogWriters() {
		if( mLogWriter != null ) {
			mLogWriter.flush();
			mLogWriter.close();
		}
		if( mSummaryWriter != null ) {
			mSummaryWriter.flush();
			mSummaryWriter.close();
		}
	}
	
}
