package edu.mayo.bior.buildcatalog;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.bior.pipeline.createcatalog.TjsonToCatalog;
import edu.mayo.bior.util.validation.CatalogFileValidator;
import org.apache.log4j.Logger;

public class CreateStep implements BuildCatalogStepInterface
{
   private static Logger sLogger = Logger.getLogger(CreateStep.class);

   private static final String STEP_DISPLAY_NAME = BuildStepKey.MAKE_CATALOG.getDisplayName();

   private static final String TAB = "\t";

   private BuildInfo mBuildInfo;
   private StepLogger mStepLogger;

   private String mTjsonInputFilePath = null;
   private String mOutputPath = null;
   private String mCatalogOutputBgzFilePath = null;
   private boolean mIsSortCatalogOnCreate = false;
   private String mChrSortOrderFilePath = null;
   private Integer mTjsonCol = -1;    // zero-based
   private String mTempDirPath = null;
   private boolean mIsInputFileAlreadyInFinalCtgFormatAndBgzipped = false;

   private String input_buildAssembly = null;
   private HumanBuildAssembly mBuildAssembly = null;

   private boolean mNormalizeInputChromosomes = false;

   public CreateStep(BuildInfo buildInfo, StepLogger stepLogger)
   {
      mBuildInfo = buildInfo;
      mStepLogger = stepLogger;
      init(mBuildInfo);
   }

   public void execute() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException
   {

      mStepLogger.log(String.format("%s step: Checking inputs...", STEP_DISPLAY_NAME));
      checkInput();

      mStepLogger.logAndSummary(String.format("Creating catalog from JSON in last column of: '%s'", mTjsonInputFilePath));
      // System.out.println("Create catalog step: creating catalog from TJSON: " + mTjsonInputFilePath);
      TjsonToCatalog tjsonToCatalog = new TjsonToCatalog();
      try
      {
         tjsonToCatalog.createCatalog(
            mTjsonInputFilePath,
            mCatalogOutputBgzFilePath,
            mIsSortCatalogOnCreate,
            mChrSortOrderFilePath,
            mTjsonCol,
            // if the tjson column is first column, this should be reasonable indicator that it is a jsonOnly TJSON file.
            mTjsonCol == 0,
            mTempDirPath,
            mBuildAssembly,
            mNormalizeInputChromosomes,
            mIsInputFileAlreadyInFinalCtgFormatAndBgzipped
         );

         File createdCatalog = new File(mCatalogOutputBgzFilePath);
         if (createdCatalog.exists())
         {
            mStepLogger.logAndSummary("Catalog created: " + mCatalogOutputBgzFilePath);
            //System.out.println("Catalog created: " + mCatalogOutputBgzFilePath);
         }

         String tabixIndexFile = mCatalogOutputBgzFilePath + CatalogMetadataConstant.TABIX_INDEX_SUFFIX;
         File createdCatalogIndex = new File(tabixIndexFile);
         if (createdCatalogIndex.exists())
         {
            mStepLogger.logAndSummary("Tabix index created: " + tabixIndexFile);
         }

      }
      catch (Exception e)
      {
         mStepLogger.logAndSummary("Error while building catalog: " + e.getMessage());
         throw new BuildCatalogStepExecuteException(e);
      }

      mStepLogger.log("Create catalog step: checking outputs ... " + mTjsonInputFilePath);
      checkOutput();

   }

   // TODO - what is this about?
   public String getLog() throws Exception
   {
      return null;
   }

   private void init(BuildInfo buildInfo)
   {
      // Right now, I believe it is assumed the catalog output would go into the
      // current working directory, no matter if the compile output is a full path
      // to somewhere else.
      mTjsonInputFilePath = buildInfo.getMakeJsonOutputFilePath();

      mOutputPath = buildInfo.getTargetDirectory();

      // The default is -1 which means last input column of TJSON input file. If the JSON string
      // to use to create the catalog is not in the last input column, pass the one-based column
      // number of the JSON column to use to build the catalog.
      mTjsonCol = buildInfo.getinputJSONColumnNum();

      // Default is to sort input rows by chromosome and position (if positional catalog).
      // Setting this to false is equivalent to the -n, --noSort bior_create_catalog
      //    command parameter to keep the input row order and not sort by BIOR chromosome sort order logic.
      mIsSortCatalogOnCreate = buildInfo.getSortCatalogOnCreate();

      // Maybe there will be a standard chromosome order file for human within build info file.
      // We don't know yet. If this is null, we'd use the default sort order implemented by the
      // existing bior_create_catalog command implementation.
      mChrSortOrderFilePath = buildInfo.getChromosomeSortOrderFileName();

      // tjsonCol==0;  // boolean indicating if catalog is json only

      mTempDirPath = buildInfo.getBaseTempDirectory();


      mStepLogger.log("CreateStep.init(): TEMP_DIR value: " + mTempDirPath);

      // just take value, don't verify it yet.
      String input_buildAssembly = buildInfo.getDataSourceBuild();
      if (input_buildAssembly != null && input_buildAssembly.startsWith(HumanBuildAssembly.GRCh37.name()))
      {
         mBuildAssembly = HumanBuildAssembly.GRCh37;
      }
      else if (input_buildAssembly != null && input_buildAssembly.startsWith(HumanBuildAssembly.GRCh38.name()))
      {
         mBuildAssembly = HumanBuildAssembly.GRCh38;
      }

      // Default is to normalize input chromosome values.
      // Set this to false, equivalent to the -k, --keepChromSame bior_create_catalog
      //    command parameter to keep the input chromosome values and not convert them.
      mNormalizeInputChromosomes = true;
      
      mIsInputFileAlreadyInFinalCtgFormatAndBgzipped = buildInfo.getInFinalFormat();
   }

   private void checkInput() throws BuildCatalogStepInputException
   {
      try
      {
         mStepLogger.log("CreateStep.checkInput(): Creating TEMP_DIR value: " + mTempDirPath);
         if (mTempDirPath != null)
         {
            File tempdir = new File(mTempDirPath);
            if (!tempdir.exists())
            {
               mStepLogger.log("CreateStep.checkInput(): Creating TEMP_DIR value: " + mTempDirPath);
               tempdir.mkdirs();
            }
         }
      }
      catch (Exception io)
      {
         // TjsonToCatalog fails without a valid temp dir passed so for now,
         // we need to fail if we can't create it, if it doesn't exist.
         throw new BuildCatalogStepInputException("CreateStep.checkInput(): Failed creating specified TEMP_DIR value: " + mTempDirPath);
      }

      try
      {
         File inputTjsonFile = null;
         if (mTjsonInputFilePath == null)
         {
            throw new BuildCatalogStepInputException("CreateStep.checkInput(): Input TJSON file value is null.");
         }
         else if (mTjsonInputFilePath.length() == 0)
         {
            throw new BuildCatalogStepInputException("CreateStep.checkInput(): Input TJSON file value is an empty string.");
         }
         else
         {
            inputTjsonFile = new File(mTjsonInputFilePath);
            if (!inputTjsonFile.canRead())
            {   // also checks existence
               throw new BuildCatalogStepInputException(String.format("CreateStep.checkInput(): Input TJSON file '%s' does not exist or cannot be read.", mTjsonInputFilePath));
            }

            verifyTJSON(inputTjsonFile);
         }
      }
      catch (Exception e)
      {
         throw new BuildCatalogStepInputException("CreateStep.checkInput(): Exception raised checking input: " + e.getMessage());
      }

      try
      {
         if (mOutputPath == null || mOutputPath.length() <= 0)
         {
            mOutputPath = new File(".").getCanonicalPath();
            System.err.println("No output directory specified.  Defaulting to the current directory: " + mOutputPath);
         }
         File outDir = new File(mOutputPath);
         if (!outDir.isDirectory())
         {
            throw new BuildCatalogStepInputException("Output directory specified [" + mOutputPath + "] does not exist or is not a directory. Please create the directory, check the build info TARGET_DIR property, and retry.");
         }
         else if (!outDir.canWrite())
         {
            throw new BuildCatalogStepInputException("Output directory specified [" + mOutputPath + "] is not writable. Please modify the permissions on this directory and retry.");
         }
      }
      catch (IOException io)
      {
         throw new BuildCatalogStepInputException("IO Exception raised while checking output path specified. Exception: " + io.getMessage());
      }

      try
      {
         mCatalogOutputBgzFilePath = mOutputPath + "/" + mBuildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
         mStepLogger.log("Catalog file path we will be creating: " + new File(mCatalogOutputBgzFilePath).getCanonicalPath());
      }
      catch (IOException io)
      {
         throw new BuildCatalogStepInputException("CreateStep.init(): Failed getting path for catalog to create.");
      }

      File outputCatFile = new File(mCatalogOutputBgzFilePath);
      if (!CatalogFileValidator.hasExpectedSuffix(mCatalogOutputBgzFilePath))
      {
         throw new BuildCatalogStepInputException("CreateStep.checkInput(): Catalog file is not a valid file. Does not have expected catalog file name suffix.");
      }

      if (outputCatFile.exists())
      {
         if (outputCatFile.canWrite())
         {
            mStepLogger.log("WARNING: CreateStep.checkInput(): Output catalog file specified already exists. It will be overwritten.");
         }
         else
         {
            throw new BuildCatalogStepInputException("ERROR: CreateStep.checkInput(): Output catalog file specified already exists. It is not writable so cannot be overwritten.");
         }
      }

   }

   private void checkOutput() throws BuildCatalogStepExecuteException
   {
      if (!CatalogFileValidator.catalogExists(mCatalogOutputBgzFilePath))
      {
         throw new BuildCatalogStepExecuteException("CreateStep.checkOutput(): Output catalog file is not a valid file. Please check catalog-build create-catalog step results.");
      }
      if (!CatalogFileValidator.hasExpectedSuffix(mCatalogOutputBgzFilePath))
      {
         throw new BuildCatalogStepExecuteException("CreateStep.checkOutput(): Output catalog file is not a valid file [" + mCatalogOutputBgzFilePath + "]. Does not have expected file suffix of [" + CatalogMetadataConstant.CATALOG_FILE_SUFFIX + "].");
      }

      try
      {
         sLogger.info("Generated output catalog: " + new File(mCatalogOutputBgzFilePath).getCanonicalPath());
      }
      catch (IOException io)
      {
         throw new BuildCatalogStepExecuteException("Unable to report canonical path of catalog output file for value: " + mCatalogOutputBgzFilePath);
      }
   }

   private void verifyTJSON(File tjsonInFile) throws BuildCatalogStepInputException
   {
      BufferedReader rdr = null;
      try
      {
         rdr = TjsonToCatalog.getBufferedReader(tjsonInFile);
         String ln = null;
         Integer previousLineElemSize = null;

         while ((ln = rdr.readLine()) != null)  {
            if (ln.startsWith("#"))  {
               continue;
            }
            
            // just white space (spaces or tabs); skip them.
            if (ln.trim().length() == 0) {
               continue;
            }

            String[] lineElements = ln.split(TAB);
            if (lineElements == null) {
               System.err.println("WARNING: CreateStep.verifyTJSON(): Please check input TJSON file. The current line had a null tab-delimited element list so skipping it: " + ln);
               continue;
            }

            // check every line to ensure user-specified JSON column is actually JSON, the default
            // of last column is JSON:
            if (mTjsonCol != -1) {
               if ((mTjsonCol - 1) < lineElements.length) {
            	   // mTjsonCol is one based, so subtract one to look in array correctly.
            	   if (!isTJsonString(lineElements[mTjsonCol - 1])) {
            		   rdr.close();
            		   throw new BuildCatalogStepInputException("User-specified JSON input column number [" + mTjsonCol.intValue() + "] in TJSON input file is not a JSON column. Column number should be the one-based column number of the JSON column that should be used to build the catalog. Column element is not JSON: [" + lineElements[mTjsonCol - 1] + "]");
            	   }
               } else {
                   rdr.close();
                   throw new BuildCatalogStepInputException("User-specified JSON input column number [" + mTjsonCol.intValue() + "] in TJSON input file is not a valid column number. Column number should be the one-based column number of the JSON column that should be used to build the catalog.");
               }
            } else if (mTjsonCol == -1 && !isTJsonString(lineElements[lineElements.length - 1])) {
               rdr.close();
               throw new BuildCatalogStepInputException("There is no user-specified JSON input column number for TJSON input file but last column in the input is not a JSON string. Column element is not JSON: [" + lineElements[lineElements.length - 1] + "]");
            }

            if (lineElements.length >= 1) {
               // first row in input:
               if (previousLineElemSize == null) {
                  previousLineElemSize = lineElements.length;
                  continue;
               }

               // if subsequent lines don't have same number elements as first line, this is bad:
               if (previousLineElemSize != lineElements.length) {
                  rdr.close();
                  throw new BuildCatalogStepInputException("Input TJSON file has inconsistent number of elements between previous line [" + previousLineElemSize.intValue() + "] and current line [" + lineElements.length + "].  current line content: " + ln);
               }
               previousLineElemSize = lineElements.length;
            } else {
               System.err.println("Skipping input line with white space: [" + ln + "].");
            }
         } // end while

      }
      catch (Exception e) {
         throw new BuildCatalogStepInputException("Exception occurred while verifying TJSON input file. Exception: " + e.getMessage());
      } finally {
         try {
            if (rdr != null) {
               rdr.close();
            }
         } catch (Exception e) {
         }
      }
   }

   private boolean isTJsonString(String s)
   {
      if (s.startsWith("{") && s.endsWith("}"))
      {
         return true;
      }
      return false;
   }
}
