package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;

import edu.mayo.cli.InvalidDataException;

/**
 * Build a full catalog with all properties files, indexes, help docs, verification of catalogs, summary of build process, etc.
 * Directory structure to build:
 * build/
 *   |- compile.sh			(compile script)
 *   |- build_info.txt
 *   |- columns.tsv.default
 *   |- datasource.properties.default
 *   |- blacklist.default
 *   |- help.txt
 *   |- help.markdown
 *   |- progress/
 *       |- MAKE_JSON.log		(successful)
 *       |- MAKE_JSON.summary	(successful)
 *       |- CREATE_CATALOG.log		(successful)
 *       |- CREATE_CATALOG.summary	(successful)
 *       |- PROP_FILES.error.log		(FAILED)
 *       |- PROP_FILES.error.summary	(FAILED)
 */
public class BuildCatalog
{
	/** The $TARGET/build subdirectory */
	public static final String BUILD_SUBDIR = "build";

    public static final String STEP_SEPARATOR_LINE = "-------------------------------------";

    public void build(String canonicalPath, BuildStepKey startStep) throws Exception {
    	BuildStepKey lastStep = BuildStepKey.values()[BuildStepKey.values().length-1];
	    build(canonicalPath, startStep, lastStep);
	}
   
   /**
    * Build catalog.  Load the BuildInfo from
    *
    * @param buildInfoPath   path to build info property file
    * @param startStep start the build with this step
    * @param endStep end the build after this step
    * @throws IOException
    * @throws InvalidDataException
    */
   public void build(String buildInfoPath, BuildStepKey startStep, BuildStepKey endStep) throws Exception
   {
      BuildInfo buildInfo = new BuildInfo(buildInfoPath);
      printBuildInfoVariables(buildInfo);

      File targetDir = new File(buildInfo.getTargetDirectory());

      if (startStep.equals(BuildStepKey.AUTO))
         startStep = determineAutoResumeStep(targetDir);

      deleteAllSuccessFilesOnAndAfterStep(startStep, targetDir);
      
      // Execute all steps that occur after stepToStartFrom
      List<BuildStepKey> stepsToExecute = getStepsToExecute(startStep, endStep);
      for(BuildStepKey stepKey : stepsToExecute)
      {
    	  String stepDisplayName = stepKey.getDisplayName();
    	  StepLogger logger = new StepLogger(stepDisplayName, targetDir);
    	  BuildCatalogStepInterface buildStep = createStepForKey(stepKey, buildInfo, logger);
    	  try {
    		  logger.logAndSummary(String.format("Step %s STARTED", stepDisplayName));
    		  buildStep.execute();
    		  markStepSuccessful(stepDisplayName, targetDir);
    		  logger.logAndSummary(String.format("Step %s SUCCEEDED", stepDisplayName));
    		  logger.summary(STEP_SEPARATOR_LINE);
    	  }
    	  catch (Exception e) {
    		  logger.log(ExceptionUtils.getStackTrace(e));
    		  System.err.println("Unexpected error executing step " + stepDisplayName + ".\n" + e.getMessage());
    		  throw e;
    	  }
    	  finally {
    		  logger.closeLogWriters();
    	  }
      }
   }


   private List<BuildStepKey> getStepsToExecute(BuildStepKey startStep, BuildStepKey endStep) {
	   boolean isAdd = false;
	   List<BuildStepKey> steps = new ArrayList<BuildStepKey>();
	   for(BuildStepKey step : BuildStepKey.values()) {
		   if( step.equals(startStep) )
			   isAdd = true;
		   if(isAdd)
			   steps.add(step);
		   if( step.equals(endStep) )
			   break;
	   }
	   return steps;
   }

   private void printBuildInfoVariables(BuildInfo buildInfo) {
	   Map<String,String> keyValPairs = buildInfo.getMap();

	   // Add BIOR_LITE_HOME, JAVA_HOME, and PATH to the map
	   keyValPairs.put("BIOR_LITE_HOME", System.getenv("BIOR_LITE_HOME"));
	   keyValPairs.put("JAVA_HOME", System.getenv("JAVA_HOME"));
	   keyValPairs.put("PATH", System.getenv("PATH"));
	   try {
		   keyValPairs.put("HOSTNAME",  InetAddress.getLocalHost().getHostName());
	   } catch(Exception e) {
		   System.err.println("Could not determine the hostname");
	   }
	   // Get the longest key so we can put into nice columns
	   int longestKeyLen = getLongestKeyLength(new ArrayList<String>(keyValPairs.keySet()));
	
	   // Print all key-value pairs
	   System.out.println();
	   System.out.println("-------------------------------------------------------------------------------------");
	   System.out.println("Variables from build_info.txt (as well as BIOR_LITE_HOME, JAVA_HOME, and PATH)");
	   System.out.println("-------------------------------------------------------------------------------------");
	   for(String key : keyValPairs.keySet()) {
		   System.out.println(StringUtils.rightPad(key, longestKeyLen + 2) + keyValPairs.get(key));
	   }
	   System.out.println("-------------------------------------------------------------------------------------");
	   System.out.println();
	   
   }



   private int getLongestKeyLength(ArrayList<String> keys) {
	   int longestLen = 0;
	   for(String key : keys) {
		   if( key.length() > longestLen ) {
			   longestLen = key.length();
		   }
	   }
	   return longestLen;
   }


   /**
    * In case the user decided to restart at a previous step, we want to clear any [step].successful
    * files at and after this step to avoid any confusion about which steps actually completed
    */
   private void deleteAllSuccessFilesOnAndAfterStep(BuildStepKey stepToStartFrom, File targetDir)
   {
      for (BuildStepKey stepKey : BuildStepKey.values())
      {
         if (stepKey.isExecuteStep(stepToStartFrom))
            getStepSuccessfulFile(stepKey.getDisplayName(), targetDir).delete();
      }
   }


   private void markStepSuccessful(String stepDisplayName, File targetDir) throws IOException
   {
      File successFile = getStepSuccessfulFile(stepDisplayName, targetDir);
      successFile.createNewFile();
   }


   private File getStepSuccessfulFile(String stepDisplayName, File targetDir)
   {
      File dir = new File(targetDir, StepLogger.PROGRESS_SUBDIR);
      return new File(dir, stepDisplayName + ".successful");
   }


   private BuildCatalogStepInterface createStepForKey(BuildStepKey step, BuildInfo buildInfo, StepLogger logger)
      throws BuildCatalogStepInputException, IOException
   {

      if (step.equals(BuildStepKey.MAKE_JSON))
      {
         return new MakeJsonStep(buildInfo, logger);
      }
      else if (step.equals(BuildStepKey.MAKE_CATALOG))
      {
         return new CreateStep(buildInfo, logger);
      }
      else if (step.equals(BuildStepKey.MAKE_PROP_FILES))
      {
         return new CreatePropFilesStep(buildInfo, logger);
      }
      else if (step.equals(BuildStepKey.MERGE_PROP_FILES))
      {
         return new MergeMetadataStep(buildInfo, logger);
      }
      else if (step.equals(BuildStepKey.MAKE_INDEXES))
      {
         return new IndexesStep(buildInfo, logger);
      }
      else if (step.equals(BuildStepKey.VERIFY))
      {
         return new VerifyStep(buildInfo, logger);
      }
      // TODO - leaving these out for now. Put back in when implementing these
//      else if (step.equals(BuildStepKey.HELP_DOCS))
//      {
//         return new HelpDocStep(buildInfo, logger);
//      }
//      else if (step.equals(BuildStepKey.SUMMARY))
//      {
//         return new SummaryStep(buildInfo, logger);
//      }

      // Step not valid!
      throw new BuildCatalogStepInputException("Step not valid: " + step.toString());
   }


   private BuildStepKey determineAutoResumeStep(File targetDir) throws IOException
   {
      // From the target area, determine which steps already exist, and the next step will be the one after that step
      BuildStepKey[] steps = BuildStepKey.values();
      // Skip the AUTO step since it won't have any output files
      for (int i = 1; i < steps.length; i++)
      {
         File successFile = getStepSuccessfulFile(steps[i].getDisplayName(), targetDir);
         if (!successFile.exists())
         {
            return steps[i];
         }
      }

      // No success files found, so start with first step
      return BuildStepKey.MAKE_JSON;
   }



}
