/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.util;

import static org.junit.Assert.assertArrayEquals;
import edu.mayo.bior.pipeline.Variant2JSONPipe;
import edu.mayo.genomicutils.refassembly.RefAssemblyFinder;
import java.io.File;
import java.io.IOException;

import org.junit.*;

/**
 * JUnit test for {@linkplain BioR}, so can test out {@linkplain BioR#annotate (String[][], String, Catalog[], String[][])}
 *
 * <p>@author Gregory Dougherty</p>
 */
public class BioRTestITCase
{
	private static String		gCatalogDir;
	private static Catalog[]	gCatalogs;
	
	private static final int kSkipHeader = 1;
	private static final String	kFieldsFile = "src/test/resources/testData/VariantToJson/Fields.tsv";
	
	
	/**
	 * 
	 * @throws java.lang.Exception
	 */
	@BeforeClass
	public static void setUpBeforeClass () throws Exception
	{
		gCatalogDir = Variant2JSONPipe.getCatalogPath () + "/";
		gCatalogs = BioR.loadCatalogs (new File ("src/test/resources/testData/VariantToJson/Catalogs.tsv"), gCatalogDir);
		
		// Verify each of the catalogs is present
		for (Catalog catalog : gCatalogs)
		{
			if ((catalog == null) || !(new File (catalog.getPath ())).exists ())
				System.err.println ("Error:  catalog not found: " + ((catalog == null) ? "(null)" : catalog.getPath ()));
		}
		
		// Set the reference assembly base directory 
		File	refAssemblyBaseDir = new File ("src/test/resources/ref_assembly");
//		File	refAssemblyBaseDir = new File (gCatalogDir + "/ref_assembly");
		RefAssemblyFinder.setRefAssemblyBaseDirStatic (refAssemblyBaseDir.getCanonicalPath ());
	}
	
	
	/**
	 * 
	 * @throws java.lang.Exception
	 */
	@AfterClass
	public static void tearDownAfterClass () throws Exception
	{
		// Does nothing
	}
	
	
	/**
	 * 
	 * @throws java.lang.Exception
	 */
	@Before
	public void setUp () throws Exception
	{
		// Does nothing
	}
	
	
	/**
	 * 
	 * @throws java.lang.Exception
	 */
	@After
	public void tearDown () throws Exception
	{
		// Does nothing
	}
	
	
	/**
	 * Test method for {@link edu.mayo.bior.util.BioR#annotate (java.lang.String[][], java.lang.String, 
	 * edu.mayo.bior.util.Catalog[], java.lang.String[][])}, where only passing in rsIDs
	 */
	@Test
	public void testAnnotateRsIDsOnly ()
	{
		String[][]	results = annotate ("src/test/resources/testData/VariantToJson/id.test", kFieldsFile, "GRCh37");
		String[][]	expected = BioR.loadFile (new File ("src/test/resources/testData/VariantToJson/id.test.results"));
		
		System.out.println("Expected: ===================================================");
		BioR.dump (expected);
		System.out.println("Actual:   ===================================================");
		BioR.dump (results);

		trimDoubles(expected);
		trimDoubles(results);
		assertArrayEquals (expected, results);
	}
	
	
	/**
	 * Test method for {@link edu.mayo.bior.util.BioR#annotate (java.lang.String[][], java.lang.String, 
	 * edu.mayo.bior.util.Catalog[], java.lang.String[][])}, where only passing in Chromosome and position
	 */
	@Test
	public void testAnnotateChromPosOnly ()
	{
		String[][]	results = annotate ("src/test/resources/testData/VariantToJson/ChrPos.test", kFieldsFile, "GRCh37");
		String[][]	expected = BioR.loadFile (new File ("src/test/resources/testData/VariantToJson/ChrPos.test.results"));
		
//		System.out.println("Expected: ===================================================");
//		BioR.dump (expected);
//		System.out.println("Actual:   ===================================================");
//		BioR.dump (results);

		trimDoubles(expected);
		trimDoubles(results);
		assertArrayEquals (expected, results);
	}
	
	
	/**
	 * Test method for {@link edu.mayo.bior.util.BioR#annotate (java.lang.String[][], java.lang.String, 
	 * edu.mayo.bior.util.Catalog[], java.lang.String[][])}.
	 * @throws IOException 
	 */
	@Test
	public void testAnnotate () throws IOException
	{
		String path = new File("/data5").getCanonicalPath();
		
		String[][]	results = annotate ("src/test/resources/testData/VariantToJson/bb.test.bim", kFieldsFile, "GRCh37");
		String[][]	expected = BioR.loadFile (new File ("src/test/resources/testData/VariantToJson/bimDrilledResults.txt"), kSkipHeader);

//		System.out.println("Expected: ===================================================");
//		BioR.dump (expected);
//		System.out.println("Actual:   ===================================================");
//		BioR.dump (results);
		
		trimDoubles(expected);
		trimDoubles(results);
		assertArrayEquals (expected, results);
	}
	
	
	/**
	 * Annotate the specified data returning the specified fields, using the specified build
	 * 
	 * @param inputPath		Path to data to annotate
	 * @param fieldsPath	Path to file holding the fields we want
	 * @param build			Build to use.  Determines the build for ref lookup, will verify that 
	 * all catalogs are for that build as well
	 * @return	2D array of results, one column per array.  First element of each array is name of column, 
	 * rest is results.  All columns will be same length.
	 */
	private String[][] annotate (String inputPath, String fieldsPath, String build)
	{
		String[][]	ids = BioR.loadFile (new File (inputPath));
		String[][]	fields = BioR.loadLines (new File (fieldsPath));
		
		return BioR.annotate (ids, build, gCatalogs, fields);
	}
	

	/** This is to resolve the issue where doubles are treated slightly differently in different versions of Java
	 *  Ex:	 expected:<0.009[]> but was:<0.009[0]>
	 *  So, trim all trailing 0's on double values	 */
	private void trimDoubles(String[][] cols) {
		for(int i=0; i < cols.length; i++) {
			for(int j=0; j < cols[i].length; j++) {
				String col = cols[i][j];
				if( isDouble(col) )
					cols[i][j] = "" + Double.parseDouble(col);
			}
		}
	}

	private boolean isDouble(String s) {
		try {
			Double.parseDouble(s);
			return true;
		} catch(Exception e) {
			return false;
		}
	}
}
