package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.catalog.verification.CatalogVerifier;
import edu.mayo.bior.catalog.verification.CatalogVerifier.VAL_TYPE;
import edu.mayo.bior.catalog.verification.MessageLogger;
import edu.mayo.bior.cli.cmd.CreateCatalogCommand;
import edu.mayo.bior.cli.cmd.VerifyCatalogCommand;

public class VerifyCatalogCommandITCase extends BaseFunctionalTest
{
	
	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	
	private File mTempDir;
	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		mTempDir = mTempFolder.newFolder();
	}
	
	@Test
	/** bior_verify was choking on JSON that was 3 levels deep: {'A':{'B':{'C':1}}}
	 *  Let's try four levels */
	public void testFourLevelsDeepJson() throws IOException {
		//String ctgData = "{'universe':{'galaxy':{'solarSystem':{'planet':'Earth'}}}}";
		String ctgData = concat("1",  "100",  "100",  "{'universe':{'galaxy':{'solarSystem':{'planet':'Earth'}}},'_landmark':'1','_minBP':100,'_maxBP':100}") + "\n";
		File jsonInputFile = new File(mTempDir, "my.json");
		FileUtils.write(jsonInputFile, ctgData);
		
		String colsTsv = concat("#ColumnName",  "Type",  "Count",  "Description",  "HumanReadableName") + "\n" +
				         concat("universe.galaxy.solarSystem.planet",  "String","1",      "planet",       "Planet") + "\n" +
				         concat("_landmark",    "String","1",      "Chrom",        "Chromosome") + "\n" +
				         concat("_minBP",       "Integer","1",     "Min base-pair","minBasePair") + "\n" +
				         concat("_maxBP",       "Integer","1",     "Max base-pair","maxBasePair");
		FileUtils.write(new File(mTempDir, "my.json.columns.tsv"), colsTsv);
		
		String datasourceProps = "### Datasource properties file for Catalog. Please fill in the descriptions to the keys below." + "\n" +
								 "ShortUniqueName=ESP_V2_GRCh37" + "\n" +
								 "Description=NHLBI GO Exome Sequencing Project (ESP) results to discover novel genes and mechanisms contributing to heart  lung and blood disorders" + "\n" +
								 "Source=ESP" + "\n" +
								 "Dataset=Variants" + "\n" +
								 "Version=V2" + "\n" +
								 "Build=GRCh37" + "\n" +
								 "Format=1.1.0";
		FileUtils.write(new File(mTempDir, "my.json.datasource.properties"), datasourceProps);
		
		File ctgOutFile = new File(mTempDir, "my.json.tsv.bgz");
		File logFile = new File(mTempDir, "my.log");
		CommandOutput crtCtgOut = runCmdApp(new CreateCatalogCommand(), "bior_create_catalog", "-i", jsonInputFile.getCanonicalPath(), "-o", ctgOutFile.getCanonicalPath());
		CommandOutput verifyOut = runCmdApp(new VerifyCatalogCommand(), "bior_verify_catalog", "-d", ctgOutFile.getCanonicalPath(), "-o", logFile.getCanonicalPath());
		
		assertEquals(verifyOut.stderr, verifyOut.exit, 0);
		assertEquals("", verifyOut.stderr);
		assertFalse(verifyOut.stdout.contains("ERROR: Keys in json not found in columns.tsv:"));
	}
	
	
	@Ignore
	@Test
	public void callValidator() throws IOException, InterruptedException
	{
		CommandOutput out = runCmdApp(new VerifyCatalogCommand(), "bior_verify_catalog", "-c", "O:/bsi/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v2/00-All.vcf.tsv.bgz");

		if( out.stderr != null && out.stderr.length() > 0 )
			System.out.println("STDERR:\n" + out.stderr + "\nSTDOUT:\n" + out.stdout);
	}
	
	

	@Ignore
	@Test
	public void testAgainstData5Catalog() throws Exception
	{
		// Ok, so this catalog definitely needs the b37p13 reference for the ref alleles to match!
		String catalogFilePath = "/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v2/00-All.vcf.tsv.bgz";

		//catalogFilePath = "/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh38/variants_nodups.v1/00-All.vcf.tsv.bgz";

		// catalogFilePath = "/data5/bsi/bioinf_int/s110048.catalog_building/ensembl_gtf/GRCh37/2014_02_07/hgnc/"; //ensembl_gtf_hgnc_complete_set.tsv.bgz";
		//catalogFilePath = "R:/bior/catalogValidation/testCatalogs/nonpositional/nonpositional_catalog.tsv.bgz";

		catalogFilePath = "/research/bsi/data/catalogs/bior/v1/omim/2014_01_13/genemap_GRCh37.tsv.bgz";

		catalogFilePath = "/data5/bsi/bioinf_int/s110048.catalog_building/omim/GRCh37/2016_04_27/ensembl_gtf/omim_genemap_ensembl_gtf.tsv.bgz";
		catalogFilePath = "/data5/bsi/BORA/Testing/jaj/bior/omim/omim_genemap_ensembl_gtf.tsv.bgz";

		// catalogFilePath = "R:/bior/catalogValidation/testCatalogs/truncated_dbSnp_catalog.tsv.bgz";

		CatalogVerifier catalogValidator = new CatalogVerifier();
		try
		{
			catalogValidator.verify(catalogFilePath, VAL_TYPE.STRICT, new MessageLogger(new StringWriter()));
		}
		catch (Exception e)
		{
			System.err.println("Exception raised in ValidateCatalogCommand: " + e.getMessage());
			e.printStackTrace();
			fail("Failed to verify catalog");
		}
	}
	

}
