package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.junit.Test;

import com.jayway.jsonpath.JsonPath;

import edu.mayo.bior.cli.cmd.Tab2JSONCommand;

public class TabToTjsonCommandITCase extends BaseFunctionalTest
{
	@Test
	public void test() throws IOException, InterruptedException
	{
        System.out.println("Tab2JSONCommandITCase.test");
        String config =
				"1	COL1	STRING	COLUMN	.	." + "\n" +
				"2	COL2	NUMBER	COLUMN	.	." + "\n" +
				"3	COL3	BOOLEAN	COLUMN	.	." + "\n";
		
		File configFile = File.createTempFile("config", ".txt");
		configFile.deleteOnExit();
		FileUtils.write(configFile, config);
		
		String stdin = concat("#COL1", "COL2", "COL3") + "\n" +
					   concat("AAAA",  "1111", "true") + "\n" +
					   // "2" still string;   2.0 -> number;   0 -> false;
					   concat("2",     "2.0",    "0") + "\n" +
					   // "3.4" still string;   0.003 -> number;   1 -> true;   
					   concat("3.4",   "0.33",   "1") + "\n" +
					   // "null" for boolean should be removed  
					   concat("D",     "99",     "null") + "\n" +
					   // "." for boolean should be removed  
					   concat("E",     "1",      ".") + "\n";

		CommandOutput out = executeScript("bior_tab_to_tjson", stdin, "--config", configFile.getAbsolutePath());

		assertEquals("STDERR:"+out.stderr+"\n"+"STDOUT:"+out.stdout, 0, out.exit);
		assertEquals("", out.stderr);

		String expected = 
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">\n" +
				concat("#COL1", "COL2", "COL3", "bior.ToTJson") + "\n" +
				concat("AAAA",  "1111", "true", "{\"COL1\":\"AAAA\",\"COL2\":1111,\"COL3\":true}") + "\n" +
				concat("2",     "2.0",  "0",    "{\"COL1\":\"2\",\"COL2\":2.0,\"COL3\":false}") + "\n" +
				concat("3.4",   "0.33", "1",    "{\"COL1\":\"3.4\",\"COL2\":0.33,\"COL3\":true}") + "\n" +
				concat("D",     "99",   "null", "{\"COL1\":\"D\",\"COL2\":99}") + "\n" +
				concat("E",     "1",    ".",    "{\"COL1\":\"E\",\"COL2\":1}") + "\n";
		assertEquals(expected, out.stdout);
		
		// Get the JSON column (3rd row, 4th col)
		String json = out.stdout.split("\n")[2].split("\t")[3];
		// Verify the type and value from each field in the JSON
        assertEquals(new String("AAAA"),	JsonPath.compile("COL1").read(json));
        assertEquals(new Integer("1111"),	JsonPath.compile("COL2").read(json));
        assertEquals(new Boolean(true),		JsonPath.compile("COL3").read(json));
	}
	
	
	@Test
	public void testLargeFloats() throws IOException, InterruptedException
	{
        System.out.println("Tab2JSONCommandITCase.test");
        //                     Col     Key       Type       InjType   Delim  Golden
        String config =	concat("1",   "COL1",   "NUMBER",   "COLUMN",  ".",   ".") + "\n";
		
		File configFile = File.createTempFile("config", ".txt");
		configFile.deleteOnExit();
		FileUtils.write(configFile, config);
		
		String stdin = "#COL1" + "\n"
				     + "123.0" + "\n"					// Double
				     + "1234567.0" + "\n"				// Double
				     + "12345678901234567890.0" + "\n" 	// BigDecimal
				     + "1234567.123456789" + "\n"		// Double
				     + "1234567.12345678901" + "\n"		// BigDecimal
				     + "0.1234567890123456" + "\n"		// Double
				     + "1.1234567890123456" + "\n"		// BigDecimal     //NOTE: This one seems to have problems because it thinks it should be a Double even though the last digit is rounded!!!!!
				     + "1.12345678901234567" + "\n"		// BigDecimal
				     + "12345678901234567890123456789.12345678901234567890123456789" + "\n";

		//CommandOutput out = executeScript("bior_tab_to_tjson", stdin, "--config", configFile.getAbsolutePath());
		setStdin(stdin);
		CommandOutput out = BaseFunctionalTest.runCmdApp(new Tab2JSONCommand(), "bior_tab_to_tjson", "--config", configFile.getCanonicalPath());

		assertEquals("STDERR:"+out.stderr+"\n"+"STDOUT:"+out.stdout, 0, out.exit);
		assertEquals("", out.stderr);

		final String EXPECTED = 
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">\n" +
				concat("#COL1",     					"bior.ToTJson") + "\n" +
				concat("123.0", 						"{\"COL1\":123.0}") + "\n" +
				concat("1234567.0", 					"{\"COL1\":1234567.0}") + "\n" +
				concat("12345678901234567890.0", 		"{\"COL1\":12345678901234567890.0}") + "\n" +
				concat("1234567.123456789", 			"{\"COL1\":1234567.123456789}") + "\n" +
				concat("1234567.12345678901", 			"{\"COL1\":1234567.12345678901}") + "\n" +
				concat("0.1234567890123456", 			"{\"COL1\":0.1234567890123456}") + "\n" +
				concat("1.1234567890123456", 			"{\"COL1\":1.1234567890123456}") + "\n" +
				concat("1.12345678901234567", 			"{\"COL1\":1.12345678901234567}") + "\n" +
				concat("12345678901234567890123456789.12345678901234567890123456789",	"{\"COL1\":12345678901234567890123456789.12345678901234567890123456789}") + "\n";
		assertEquals(EXPECTED, out.stdout);
	}
}
