package edu.mayo.bior.cli.cmd;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.escapeChrs;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import edu.mayo.bior.catalog.markdown.transformer.TransformException;
import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.markdown.MarkdownInfo;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;

public class CatalogMarkdownCommandTest  extends BaseFunctionalTest {

	
	private CatalogMarkdownCommand mCtgMarkdown = new CatalogMarkdownCommand();
	
	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	private File mTempDir = null;
	
	private File catalogDir; 
	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		mTempDir = mTempFolder.newFolder();
		
		this.catalogDir = mTempFolder.newFolder("Catalog");
		
		// Copy one of the test catalogs to the catalog dir
		FileUtils.copyDirectory(new File("src/test/resources/metadata"), this.catalogDir);
		
		// Modify the datasource.properties file to have a "Format=1.1.0" line
		File datasrcProps = new File(this.catalogDir, "00-All_GRCh37.datasource.properties");
		String datasrcStr = FileUtils.readFileToString(datasrcProps) + "\nFormat=1.1.0\n";
		FileUtils.write(datasrcProps, datasrcStr);
		
		// Create a build directory and in it put a build_info.txt file with the first line being 
		// MAKE_JSON_ARGS=/data5/bsi/refdata-new/biobase/human/variant/hgmd/latest/downloaded/2017_01_06  rafaib
		// The first param of which will be used to find the refdata source
		File buildDir = new File(this.catalogDir, "build");
		buildDir.mkdirs();
		String buildInfoStr = "MAKE_JSON_ARGS=/data5/bsi/refdata-new/biobase/human/variant/hgmd/latest/downloaded/2017_01_06  rafaib";
		FileUtils.write(new File(buildDir, "build_info.txt"), buildInfoStr);
	}
	
	@Test
	/** Test the cmd line call with all parameters */
	public void cmdLine() throws UnsupportedEncodingException, IOException {
		File outputDir = mTempFolder.newFolder();
		File markdownFile = new File(outputDir, "00-All_GRCh37.md");
		CommandOutput out = runCmdApp(new CatalogMarkdownCommand(), "bior_catalog_markdown",
				"-d", new File(this.catalogDir, "00-All_GRCh37.tsv.bgz").getCanonicalPath(),
				"-o", outputDir.getCanonicalPath());
		// If cmd ran successfully it should have no output to either stdout or stderr
		assertEquals("", out.stderr);
		assertEquals(0, out.exit);
		assertEquals("", out.stdout);
		
		String markdownOutput = FileUtils.readFileToString(markdownFile);
		assertBasicMarkdownContents(markdownOutput);
	}

	
	@Test
	public void testOutputDir_defaultsToCurrentWorkingDir() throws IOException, CatalogFormatException {
		//setCurrentWorkingDirectory(mTempDir.getCanonicalPath());
		CommandOutput out = runCmdApp(new CatalogMarkdownCommand(), "bior_catalog_markdown",
				"-d", new File(this.catalogDir, "00-All_GRCh37.tsv.bgz").getCanonicalPath()
				);
		// If cmd ran successfully it should have no output to either stdout or stderr
		assertEquals("", out.stderr);
		assertEquals(0, out.exit);
		assertEquals("", out.stdout);
		
		File markdownFile = new File(".", "00-All_GRCh37.md");
		System.out.println("markdown path: " + markdownFile.getCanonicalPath());
		markdownFile.deleteOnExit();
		String markdownOutput = FileUtils.readFileToString(markdownFile);
		assertBasicMarkdownContents(markdownOutput);
	}

	// NOTE: This doesn't seem to work to set the current working directory for tests
	public static boolean setCurrentWorkingDirectory(String directory_name)	{
		// Desired current working directory
		File  currentWorkingDirectory = new File(directory_name).getAbsoluteFile();
		if (currentWorkingDirectory.exists() || currentWorkingDirectory.mkdirs()) {
			// Return true if successful, else false
			return System.setProperty("user.dir", currentWorkingDirectory.getAbsolutePath()) != null;
		}
		// Directory does not exist or was not created - fail
		return false;
	}
	
	@Test
	public void goodPathWithDeprecation() throws IOException, CatalogFormatException, TransformException {
		//File ctg = new File("/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v1/00-All.vcf.tsv.bgz");
		//File ctg = new File("/research/bsi/data/catalogs/bior/v1/HGMD/2016.4_GRCh37/variants.v1/HGMD_PRO_2016.4.vcf.tsv.bgz");
		File ctg = new File(this.catalogDir, "00-All_GRCh37.tsv.bgz");
		// General output folder - the markdown folder will be created here
		File outputDir = mTempFolder.newFolder();
		MarkdownInfo markdown = mCtgMarkdown.createMarkdown(ctg, outputDir, /*previousMarkdown=*/null, null);
		assertBasicMarkdownContents(markdown.markdownOutputStr);
	}
	
	
	private void assertBasicMarkdownContents(String markdownOutputStr) {
		assertTrue(markdownOutputStr.startsWith("# ["));
		assertContains(markdownOutputStr, "#### Purpose:");
		assertContains(markdownOutputStr, "#### Citation(s):");
		assertContains(markdownOutputStr, "## Technical Specifications:");
		assertContains(markdownOutputStr, "### Properties File:\n"
												+  "[//]: # (NOTE: This is auto-generated from the <CATALOG>.datasource.properties file)\n"
												+  "Property | Values");
		assertContains(markdownOutputStr, "### Content Details:\n"
												+  "[//]: # (NOTE: This is auto-generated from the <CATALOG>.columns.tsv file)\n"
												+  "ColumnName | Type | Count | Description | HumanReadableName\n"
												+  "------------ | ------------ | ------------ | ------------ | ------------\n"
												+  "\\_altAlleles | String | \\. | One or more alternate alleles \\(non\\-reference\\) in a JSON array \\(basically a comma\\-separated list\\) \\(BioR field\\) | "); //\\_altAlleles"); //Alternate alleles \\(BioR\\)\n");
		assertContains(markdownOutputStr, "#### Locations:\n"
												+  "Name | Path on RCF\n"
												+  "---- | ----\n"
												+  "Catalog Path | ");		
	}
}
