package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.cli.func.BaseFunctionalTest;

public class MergeMetadataStepTest
{

   MergeMetadataStep mergeStep = null;
   File targetDir = null;
   Map<String, String> buildInfoMap = new HashMap<String, String>();
   BuildInfo buildInfo = null;
   StepLogger stepLogger = null;
   TemporaryFolder tempFolder;
   File previousCtgDir;
   
   public static final String BLACKLIST_SUBHEADER = 
		   "### If this file is NOT present, then assume that all fields, except for the golden attributes (those beginning with _) should be shown.\n" +
		   "### If this file is present, but contains no columns, then assume that all fields should be shown.\n";
   
   public static final String BLACKLIST_HEADER =  
   		   "### These columns are to generally be ignored within UIs as they are duplicates or are normalized versions of the original columns.\n" +
   		   BLACKLIST_SUBHEADER;
   
   public static final String BLACKLIST_BIORWEB_HEADER = 
		   "### These fields will NOT be shown in BioRWeb.  All others will." + "\n" +
		   BLACKLIST_SUBHEADER;
   
   public static final String COLUMNS_TSV_HEADER = 
		  "##-----------------------------------------------------\n" +
		  "## Catalog field definitions\n" +
		  "##-----------------------------------------------------\n" +
		  "## ColumnName = The key or column name\n" +
		  "## Type = The type of the object, as can be determined from parsing the VCF file or taking an educated guess based on the catalog values (Possible values: JSON, String, Integer, Float, Boolean)\n" +
		  "## Count = The number of values that repeatedly occur  (Possible values: 0 (Boolean), 1 (JSON,Integer,Float,String), or '.' for variable or unknown number of values.  JsonArrays will be resolved to other primitives such as String,Integer,Float, and will have a count of '.'\n" +
		  "## Description = The description of the ColumnName\n" +
		  "## HumanReadableName = A more readable name than ColumnName but typically not as long as Description.\n" +
		  "##------------------------------------------------------------------\n" +
		  BaseFunctionalTest.concat("#ColumnName",  "Type",   "Count", "Description",                     "HumanReadableName") + "\n";

   
   private final String EOL = "\n";


   @Before
   public void beforeEach() throws IOException, BuildCatalogStepInputException
   {
      tempFolder = new TemporaryFolder();
      tempFolder.create();
      targetDir = tempFolder.newFolder();
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), 			targetDir.getCanonicalPath());
      buildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), 		"x");
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), 		"");
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), "2");
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(), 		"name");
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(),	"");
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), new File("src/test/resources/buildCatalog/makeJsonStep.sh").getCanonicalPath());
      buildInfo = new BuildInfo(buildInfoMap);
      stepLogger = new StepLogger(BuildStepKey.MERGE_PROP_FILES.toString(), targetDir);
   }

   private void copyPropFilesToTemp() throws IOException
   {
      FileUtils.copyDirectory(new File("src/test/resources/buildCatalog/mergeProps"), targetDir);
   }

   private void copyPreviousCatalogFiles() throws IOException, BuildCatalogStepInputException
   {
      String fromDir = "src/test/resources/buildCatalog/mergeProps_previousCatalog";
      previousCtgDir = tempFolder.newFolder();

      // columns tsv from src/test/resources to previous catalog dir
      FileUtils.copyFile(
    		  new File(fromDir, 		"x.columns.tsv"),
    		  new File(previousCtgDir,  "x.columns.tsv")
    		  );

      // datasource.properties to previous catalog dir
      FileUtils.copyFile(
    		  new File(fromDir, 		"x.datasource.properties"),
    		  new File(previousCtgDir, 	"x.datasource.properties")
    		  );

      // Catalog file
      FileUtils.copyFile(
    		  new File(fromDir, 		"x.tsv.bgz"),
    		  new File(previousCtgDir, 	"x.tsv.bgz")
    		  );

      // Blacklist file
      FileUtils.copyFile(
    		  new File(fromDir,			"x.columns.tsv.blacklist"),
    		  new File(previousCtgDir,	"x.columns.tsv.blacklist")
    		  );

      // Blacklist (BiorWeb) file
      FileUtils.copyFile(
    		  new File(fromDir,			"x.columns.tsv.blacklist.biorweb"),
    		  new File(previousCtgDir,	"x.columns.tsv.blacklist.biorweb")
    		  );

      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), previousCtgDir.getCanonicalPath() + "/x.tsv.bgz");
      buildInfo = new BuildInfo(buildInfoMap);
   }

   @Test
   /** A clean run that has no other files in the directory - just the ones created from the CreateMetadataStep */
   public void testFirstRun() throws IOException, BuildCatalogStepExecuteException
   {
      copyPropFilesToTemp();
      mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
      mergeStep.execute();

      // Verify that all files are the same as the input since we didn't change anything
      String srcTestDir = "src/test/resources/buildCatalog/mergeProps";
      assertEquals(
    		  FileUtils.readFileToString(new File(srcTestDir, "x.columns.tsv")),
    		  FileUtils.readFileToString(new File(targetDir,  "x.columns.tsv")));
      assertEquals(
    		  FileUtils.readFileToString(new File(srcTestDir, "x.columns.tsv.blacklist")),
    		  FileUtils.readFileToString(new File(targetDir,  "x.columns.tsv.blacklist")));
      assertEquals(
    		  FileUtils.readFileToString(new File(srcTestDir, "x.columns.tsv.blacklist.biorweb")),
    		  FileUtils.readFileToString(new File(targetDir,  "x.columns.tsv.blacklist.biorweb")));
      // datasource.properties: Build is the only one not specified out of Source,Version,Build
      // NOTE: Load the input file, then modify 3 of the lines that get updated because of the previous catalog 
      String datasrcExpected = FileUtils.readFileToString(new File(srcTestDir, "x.datasource.properties"))
    		  .replaceAll("ShortUniqueName=.*", "ShortUniqueName=name_2_BUILD")
    		  .replaceAll("Source=.*",          "Source=name")
    		  .replaceAll("Version=.*",         "Version=2");
      assertEquals(
    		  datasrcExpected,
    		  FileUtils.readFileToString(new File(targetDir,  "x.datasource.properties")));
   }

   
   @Test
   public void testSuffixes() throws IOException {
	   mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
	   File ctg = new File("somedir/00-All.vcf.tsv.bgz");
	   assertEquals("00-All.vcf.columns.tsv", 					mergeStep.getColumnsTsvFromCatalog(ctg).getName());
	   assertEquals("00-All.vcf.columns.tsv.blacklist", 		mergeStep.getBlacklistFromCatalog(ctg).getName());
	   assertEquals("00-All.vcf.columns.tsv.blacklist.biorweb", mergeStep.getBlacklistBiorwebFromCatalog(ctg).getName());
   }

   @Test
   /** A clean first run, but using another catalog as a source for some values in the columns.tsv and datasource.properties */
   public void testFirstRunAgainstPreviousCatalog() throws IOException, BuildCatalogStepExecuteException,
      BuildCatalogStepInputException
   {
      copyPropFilesToTemp();
      copyPreviousCatalogFiles();

      mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
      mergeStep.execute();

      MergeColumnsTsvTest.assertDirContainsOnly(new File(buildInfo.getTargetDirectory()), Arrays.asList(
         BuildCatalog.BUILD_SUBDIR,
         "x.columns.tsv",
         "x.columns.tsv.blacklist",
         "x.columns.tsv.blacklist.biorweb",
         "x.datasource.properties"
      ));
      MergeColumnsTsvTest.assertDirContainsOnly(new File(buildInfo.getTargetDirectory(), BuildCatalog.BUILD_SUBDIR), Arrays.asList(
         "progress",
         "x.columns.tsv.01",          // NOTE: Backup
         "x.columns.tsv.default",
         "x.datasource.properties.01",// NOTE: Backup
         "x.datasource.properties.default"
      ));
      MergeColumnsTsvTest.assertDirContainsOnly(new File(buildInfo.getTargetDirectory(), BuildCatalog.BUILD_SUBDIR + "/progress"), Arrays.asList(
         BuildStepKey.MERGE_PROP_FILES.toString() + ".log",
         BuildStepKey.MERGE_PROP_FILES.toString() + ".summary"
      ));


      // Verify that all files are the same as the input since we didn't change anything
      String srcDir = "src/test/resources/buildCatalog/mergeProps";
      String srcDirWithVals = "src/test/resources/buildCatalog/mergeProps_previousCatalog";
      
      // The 3 values "ShortUniqueName", "Source", and "Version" will be replaced by values from build_info.txt
      String dataSrcExpected = FileUtils.readFileToString(new File(srcDirWithVals, "x.datasource.properties"))
    		  .replace("_CATALOG_PREFIX_", "x")  // This occurs in the comment at the top
    		  .replaceAll("ShortUniqueName=.*",   "ShortUniqueName=name_2_BUILD")
    		  .replaceAll("Source=.*",            "Source=name")
    		  .replaceAll("Version=.*",           "Version=2");
      
      assertEquals(FileUtils.readFileToString(new File(srcDirWithVals, "x.columns.tsv")),
    		  	   FileUtils.readFileToString(new File(targetDir,      "x.columns.tsv")));
      
      
      String EXPECTED_DEFAULT = "_altAlleles" + EOL
    		  				+   "_id" + EOL
    		  				+   "_landmark" + EOL
    		  				+   "_maxBP" + EOL
    		  				+   "_minBP" + EOL
    		  				+   "_refAllele" + EOL;
    		  
      assertEquals(BLACKLIST_HEADER + EXPECTED_DEFAULT, FileUtils.readFileToString(new File(targetDir, "x.columns.tsv.blacklist")));
      
      assertEquals(BLACKLIST_BIORWEB_HEADER + EXPECTED_DEFAULT, FileUtils.readFileToString(new File(targetDir, "x.columns.tsv.blacklist.biorweb")));
      
      assertEquals(dataSrcExpected, FileUtils.readFileToString(new File(targetDir, "x.datasource.properties")));
   }

   @Test
   /** A clean first run, but using another catalog as a source for some values in the columns.tsv and datasource.properties */
   public void testRunAgainstBadPreviousCatalog() throws IOException, BuildCatalogStepExecuteException,
      BuildCatalogStepInputException
   {
      String srcDir = "src/test/resources/buildCatalog/mergeProps_previousCatalog/";
      File prevCatFrom = new File(srcDir, "/x.tsv.bgz");

      previousCtgDir = tempFolder.newFolder();
      //System.out.println("Previous catalog directory set to: " + previousCtgDir);
      String prevCatalogFilePathNameTo = previousCtgDir.getCanonicalPath() + "/x.tsv.bgz";
      File prevCatTo = new File(prevCatalogFilePathNameTo);

      // to get through this call to buildInfo, create the file
      prevCatTo.createNewFile();
      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), prevCatalogFilePathNameTo);
      buildInfo = new BuildInfo(buildInfoMap);

      // now delete before this test
      prevCatTo.delete();
      // Test: No catalog file exists at all:
      try
      {
         mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
         fail("The catalog file does not exist so this should throw an exception and not get here.");
      }
      catch (IOException io)
      {
      }

      // Test: Catalog file is not readable
      FileUtils.copyFile(prevCatFrom, prevCatTo);
      prevCatTo.setReadable(false);
      try
      {
         mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
         fail("The catalog file is not readable so this should throw an exception and not get here.");
      }
      catch (IOException io)
      {
         assertTrue(io != null);
      }

      // Test: Catalog file is readable but prop files for catalog do not exist:
      prevCatTo.setReadable(true);
      try
      {
         mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
         fail("The catalog file is readable but none of the prop files exist, so this should throw an exception and not get here.");
      }
      catch (IOException io)
      {
         assertTrue(io != null);
      }

      // Test: Catalog file exists, is readable, and at least one prop file exists so this one should succeed:
      mergeStep = null;
      File prevColsFrom = new File(srcDir, "x.columns.tsv");
      File prevColsTo = new File(previousCtgDir, "x.columns.tsv");
      FileUtils.copyFile(prevColsFrom, prevColsTo);
      prevColsTo.setReadable(true);
      try
      {
         mergeStep = new MergeMetadataStep(buildInfo, stepLogger);
         assertTrue(mergeStep != null);
      }
      catch (IOException io)
      {
         fail("The catalog file is readable but none of the prop files exist, so this should throw an exception and not get here.");
      }
   }
}
