package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import edu.mayo.cli.InvalidDataException;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

/**
 * Tests for the MAKE_JSON step within the BuildCatalog command
 *
 * @author Michael Meiners (m054457)
 *         2016-04-25
 */
public class MakeJsonStepTest
{

   private File mTargetDir = null;
   private static final String JSON_OUT_FILE_PREFIX = "jsonOutput";
   private static final String JSON_OUT_FILE_SUFFIX = ".txt";

   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   @Before
   public void beforeEach() throws IOException {
	   TemporaryFolder tempFolder = new TemporaryFolder();
	   tempFolder.create();
	   mTargetDir = tempFolder.newFolder();
   }

   @Test
   public void testValidCompile() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException,
      IOException, InvalidDataException
   {
      BuildInfo buildInfo = getBuildInfo("src/test/resources/buildCatalog/makeJsonStep.sh");
      StepLogger stepLogger = new StepLogger(BuildStepKey.MAKE_JSON.toString(), mTargetDir);
      MakeJsonStep compile = new MakeJsonStep(buildInfo, stepLogger);
      compile.execute();
      assertEquals("", compile.getScriptStderr());
      assertTrue(compile.getScriptStdout().endsWith("Done\n"));
      assertEquals(0, compile.getScriptExitCode());
   }

   @Test
   public void testValidCompileSetEnvOKCompileOutputFilePath() throws BuildCatalogStepInputException,
      BuildCatalogStepExecuteException, IOException, InvalidDataException
   {
      BuildInfo buildInfo = getBuildInfo("src/test/resources/buildCatalog/makeJsonStepPrintEnvVar.sh");
      StepLogger stepLogger = new StepLogger(BuildStepKey.MAKE_JSON.toString(), mTargetDir);
      MakeJsonStep compile = new MakeJsonStep(buildInfo, stepLogger);
      compile.execute();
      assertEquals("", compile.getScriptStderr());
      assertTrue(compile.getScriptStdout().endsWith("Done\n"));
      assertEquals(0, compile.getScriptExitCode());

      String scriptOutputPrefix = "makeJsonOutputFilePath=";
      assertTrue(compile.getScriptStdout().contains(scriptOutputPrefix));
      int indexOfOurOutput = compile.getScriptStdout().indexOf(scriptOutputPrefix);
      int indexOfNextNL = compile.getScriptStdout().indexOf("\n", indexOfOurOutput);
      String envVarPrinted = compile.getScriptStdout().substring(indexOfOurOutput + scriptOutputPrefix.length(), indexOfNextNL);
      assertTrue(envVarPrinted.contains(JSON_OUT_FILE_PREFIX));
      assertTrue(envVarPrinted.endsWith(JSON_OUT_FILE_SUFFIX));
      assertTrue(new File(envVarPrinted).exists());
   }

   @Test
   public void testCompileErrorBadScriptPath() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException,
      IOException
   {
      String scriptName = "/dirDoesNotExist/compile.sh";

      expectedException.expect(BuildCatalogStepInputException.class);
      expectedException.expectMessage("could not find script '" + scriptName);

      BuildInfo buildInfo = getBuildInfo(scriptName);
      StepLogger stepLogger = new StepLogger(BuildStepKey.MAKE_JSON.toString(), mTargetDir);
      MakeJsonStep compile = new MakeJsonStep(buildInfo, stepLogger);
      compile.execute();
   }


   @Test
   public void testCompileErrorWithBadOutput() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException,
      IOException, InvalidDataException
   {
      BuildInfo buildInfo = getBuildInfo("src/test/resources/buildCatalog/makeJsonStepBadOutput.sh");
      try
      {
         StepLogger stepLogger = new StepLogger(BuildStepKey.MAKE_JSON.toString(), mTargetDir);
         MakeJsonStep compile = new MakeJsonStep(buildInfo, stepLogger);
         compile.execute();
         fail("Should not have reached this line");
      }
      catch (Exception e)
      {
         assertTrue(e instanceof BuildCatalogStepExecuteException);
         assertEquals("Error: each line in the make_json output must end with a JSON object", e.getMessage());
      }
   }


   @Test
   public void testCompileErrorNoOutputFile() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException,
      IOException, InvalidDataException
   {
      BuildInfo buildInfo = getBuildInfo("src/test/resources/buildCatalog/makeJsonStepEmptyOutputFile.sh");

      expectedException.expect(BuildCatalogStepExecuteException.class);
      expectedException.expectMessage("Error: output file was empty: '" + buildInfo.getMakeJsonOutputFilePath());
      StepLogger stepLogger = new StepLogger(BuildStepKey.MAKE_JSON.toString(), mTargetDir);
      MakeJsonStep compile = new MakeJsonStep(buildInfo, stepLogger);
      compile.execute();
      fail("Should not have reached this line");
   }

   @Test
   public void testCompileErrorNonZeroExitCode() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException,
      IOException, InvalidDataException
   {
      BuildInfo buildInfo = getBuildInfo("src/test/resources/buildCatalog/makeJsonStepOutputNonZero.sh");

      expectedException.expect(BuildCatalogStepExecuteException.class);
      expectedException.expectMessage("was non-zero: 1");

      StepLogger stepLogger = new StepLogger(BuildStepKey.MAKE_JSON.toString(), mTargetDir);
      MakeJsonStep compile = new MakeJsonStep(buildInfo, stepLogger);
      compile.execute();
      fail("Should not have reached this line");
   }


   //======================================================================

   private BuildInfo getBuildInfo(String scriptToRun) throws IOException, BuildCatalogStepInputException
   {
      Map<String, String> map = new HashMap<String, String>();
      File compileScript = new File(scriptToRun);
      map.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), compileScript.getCanonicalPath());
      map.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile inputFile  --column 4");
      File jsonOutFile = File.createTempFile(JSON_OUT_FILE_PREFIX, JSON_OUT_FILE_SUFFIX);
      map.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), jsonOutFile.getCanonicalPath());
      map.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), "");
      map.put(BuildInfoKey.DATA_SOURCE.name(), "x");
      map.put(BuildInfoKey.CATALOG_PREFIX.name(), "x");
      map.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), "2");
      return new BuildInfo(map);
   }

}
