package edu.mayo.bior.pipeline.createcatalog;

import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;

import org.apache.log4j.Logger;
import org.json.JSONObject;

import com.tinkerpop.pipes.AbstractPipe;

import edu.mayo.pipes.bioinformatics.vocab.CoreAttributes;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.util.GenomicObjectUtils;

/**
 * Convert a tab-delimited line that contains JSON
 * Ex:  ABC	typeVariant  	1	 100  101  {"_landmark":"1","_minBP":100,"_maxBP":101,....}
 * Into a Catalog compatible line (retaining any header lines
 * Ex:  1  100  100  {"_landmark":"1"....
 */
public class TjsonToCatalogPipe extends AbstractPipe<History, History>
{
   private static Logger sLogger = Logger.getLogger(TjsonToCatalogPipe.class);

   private int mJsonCol = -1;
   private boolean mIsJsonOnly = false;

   private boolean mIsFirstRow = true;
   private boolean mIsModifyChrom = true;

   private final String LANDMARK = CoreAttributes._landmark.toString();
   private final String MINBP = CoreAttributes._minBP.toString();
   private final String MAXBP = CoreAttributes._maxBP.toString();
   private final String REFALLELE = CoreAttributes._refAllele.toString();

   // Show warnings for each field that is changed automatically
   private boolean mIsWarnedOfLandmarkNonString = false;
   private boolean mIsWarnedOfMinbpNonLong = false;
   private boolean mIsWarnedOfMaxbpNonLong = false;
   private boolean mIsWarnedOfMaxbpCalculated = false;
   private List<String> mLandmarksWarned = new ArrayList<String>();

   private String mJsonOriginal = null;

   /**
    * Constructor for TjsonToCatalogPipe
    *
    * @param jsonCol    The positive or negative index of the JSON column within the data rows
    * @param isJsonOnly Should the catalog consist of only a JSON column (and not the chrom,minBP,maxBP as well)?
    */
   public TjsonToCatalogPipe(int jsonCol, boolean isJsonOnly, boolean isModifyChromosome)
   {
      mJsonCol = jsonCol;
      mIsJsonOnly = isJsonOnly;
      mIsModifyChrom = isModifyChromosome;
   }


   @Override
   protected History processNextStart() throws NoSuchElementException
   {
      History historyIn = getNextNonBlankNonHeaderLine();

      // If it's a header line, then return it without processing
      if (isHeaderLine(historyIn))
         return historyIn;

      fixJsonColumnIfFirstDataRow(historyIn.size());

      History historyOut = processLine(historyIn);
      return historyOut;
   }


   private boolean isHeaderLine(History historyIn)
   {
      return historyIn.get(0).startsWith("#");
   }


   private History getNextNonBlankNonHeaderLine()
   {
      History historyIn = null;
      do
      {
         historyIn = this.starts.next();
      }
      while (isBlankOrHeaderLine(historyIn));

      return historyIn;
   }


   private void fixJsonColumnIfFirstDataRow(int colCount)
   {
      if (!mIsFirstRow)
         return;

      mIsFirstRow = false;

      // If the json column is 0 or higher, then subtract 1 to make the index 0-based
      if (mJsonCol > -1)
      {
         mJsonCol = mJsonCol - 1;
      }

      // If the json column is negative, then make it positive, with -1 representing the last column
      if (mJsonCol <= -1)
      {
         mJsonCol = colCount + mJsonCol;
      }
   }


   private boolean isBlankOrHeaderLine(History line)
   {
      boolean isBlank = line == null || line.size() == 0 || (line.size() == 1 && line.get(0).trim().length() == 0);
      boolean isHeader = line != null && line.size() > 0 && line.get(0).startsWith("#");
      return isBlank || isHeader;
   }

   /**
    * Process the next line and turn it into a catalog line
    *
    * @param lineIn The next tab-delimited line to convert into a catalog line
    * @return New History object containing either one new JSON column or 3 tabix columns plus JSON
    */
   public History processLine(History lineIn)
   {
      if (lineIn.size() == 0 || mJsonCol < 0 || mJsonCol > (lineIn.size() - 1))
         throw new RuntimeException("JSON column (" + mJsonCol + ") is out of range on row: " + lineIn.getMergedData("\t"));

      mJsonOriginal = lineIn.get(mJsonCol);
      // See local project for org.json.JSONObject
      JSONObject jsonObj = null;
      com.google.gson.JsonObject gsonObj = null;
      try
      {
         jsonObj = new JSONObject(mJsonOriginal);
      }
      catch (org.json.JSONException e)
      {
         throw new IllegalArgumentException("Failed to create JSONObject for TJSON input row: " + lineIn.getMergedData("\t"));
      }
      String chrom = getLandmark(jsonObj, mIsModifyChrom);

      Long minBP = getMinBP(jsonObj);

      Long maxBP = getMaxBP(jsonObj);

      if (mIsJsonOnly)
      {
         return new History(new String[]{jsonObj.toString()});
      }
      else
      {
         return new History(new String[]{chrom, minBP.toString(), maxBP.toString(), jsonObj.toString()});
      }
   }

   /**
    * Get the maxBP value, correcting it in the JSON object as necessary.
    * If _maxBP value is not given in the JSON, but _minBP and _refAllele were,
    * then calculate _maxBP based on _minBP and _refAllele length.
    *
    * @param jsonObj The JSONObject representing the JSON string to be drilled
    * @return Long value for maxBP, or 0L if not found
    */
   public Long getMaxBP(JSONObject jsonObj)
   {
      Object maxbpObj = getFromJson(jsonObj, MAXBP);

      Object minbpObj = getFromJson(jsonObj, MINBP);
      Object refObj = getFromJson(jsonObj, REFALLELE);

      // If not found, try to calculate it from the _minBP and _refAllele length.  If those aren't given, then set it to 0
      if (maxbpObj == null)
      {
         if (minbpObj == null || refObj == null)
         {
            return 0L;
         }

         if (!mIsWarnedOfMaxbpCalculated)
         {
            sLogger.warn("Warning: " + CoreAttributes._maxBP.toString()
               + " was not specified, so it will be calculated from "
               + CoreAttributes._minBP.toString() + " and " + CoreAttributes._refAllele.toString() + " length.");
            sLogger.warn("First offending JSON shown: " + jsonObj.toString());
            mIsWarnedOfMaxbpCalculated = true;
         }
         // They are present, so try to calculate:  _maxBP = (_minBP + refAllele.length() - 1)
         Long minbp = Long.parseLong(minbpObj.toString());
         String refAllele = refObj.toString();
         Long maxbp = minbp + refAllele.length() - 1;
         jsonObj.put(MAXBP, maxbp);
         return maxbp;
      }

      // If _minBP, _maxBP, and _refAllele are all specified, then throw exception if they do not add up
      if (minbpObj != null && maxbpObj != null && refObj != null)
      {
         Long minbpLong = Long.parseLong(minbpObj.toString());
         Long maxbpLong = Long.parseLong(maxbpObj.toString());
         int refLen = refObj.toString().length();
         if ((maxbpLong - minbpLong) != (refLen - 1))
            throw new IllegalArgumentException("Error:  length of the refAllele does not equal (max-min+1): " + mJsonOriginal);
      }

      // If it is NOT a Long, then convert it to a Long, and re-insert it back into the JSON as a Long
      String clsName = maxbpObj.getClass().getName();
      Long maxbp = Long.parseLong(maxbpObj.toString());
      if (!(maxbpObj instanceof Integer))
      {
         if (!mIsWarnedOfMaxbpNonLong)
         {
            sLogger.warn("Warning: " + CoreAttributes._maxBP.toString()
               + " was not an integer.  Correcting offending JSON (only first original shown): " + mJsonOriginal);
            mIsWarnedOfMaxbpNonLong = true;
         }
         jsonObj.put(MAXBP, maxbp);
      }

      return maxbp;
   }

   /**
    * Get the maxBP value, correcting it in the JSON object as necessary
    *
    * @param jsonObj The JSONObject representing the JSON string to be drilled
    * @return Long value for maxBP, or 0L if not found
    */
   public Long getMinBP(JSONObject jsonObj)
   {
      Object minbpObj = getFromJson(jsonObj, MINBP);

      // If not found, set it to 0
      if (minbpObj == null)
         return 0L;

      // If it is NOT a Long, then convert it to a Long, and re-insert it back into the JSON as a Long
      Long minbp = Long.parseLong(minbpObj.toString());
      String clsName = minbpObj.getClass().getName();
      if (!(minbpObj instanceof Integer))
      {
         if (!mIsWarnedOfMinbpNonLong)
         {
            sLogger.warn("Warning: " + CoreAttributes._minBP.toString()
               + " was not an integer.  Correcting offending JSON (only first original shown): " + mJsonOriginal);
            mIsWarnedOfMinbpNonLong = true;
         }
         jsonObj.put(MINBP, minbp);
      }

      return minbp;
   }


   /**
    * Get _landmark, and fix if necessary
    *
    * @param jsonObj       The JSONObject representing the JSON string to be drilled
    * @param isModifyChrom Whether the user chose to let the _landmark be modified if necessary
    * @return
    */
   public String getLandmark(JSONObject jsonObj, boolean isModifyChrom)
   {
      Object landmarkObj = getFromJson(jsonObj, LANDMARK);

      // If not found, set it to "UNKNOWN"
      if (landmarkObj == null)
         return "UNKNOWN";

      //String clsName = landmarkObj.getClass().getName();
      String landmark = landmarkObj.toString().trim();

      // If user has allowed us to modify the chromosome to fit Human chromosomes
      if (isModifyChrom)
      {
         // If it starts with "chr" and is longer than 3 chars, then cut off the "chr" from the front
         // Convert 23 to X,  24 to Y,  25 to XY,  26 to M,  MT to M, etc
         String landmarkModified = GenomicObjectUtils.computechr(landmark);
         if (!landmark.equals(landmarkModified))
         {
            // Warn about this landmark being changed if we have not seen it yet
            if (!mLandmarksWarned.contains(landmark))
            {
               sLogger.warn("Warning: _landmark was changed from " + landmark + " to " + landmarkModified
                  + ".  Correcting offending JSON (only first original shown): " + mJsonOriginal);
               mLandmarksWarned.add(landmark);
            }
            landmark = landmarkModified;
            jsonObj.put(LANDMARK, landmarkModified);
         }
      }

      // Landmark must be a String.
      // If it is NOT a string, then convert it to a string, and re-insert it back into the JSON as a String
      if (!(landmarkObj instanceof String))
      {
         if (!mIsWarnedOfLandmarkNonString)
         {
            sLogger.warn("Warning: " + CoreAttributes._landmark.toString()
               + " was not a String.  Correcting offending JSON (only first original shown): " + mJsonOriginal);
            mIsWarnedOfLandmarkNonString = true;
         }
         jsonObj.put(LANDMARK, landmark);
      }

      return landmark;
   }

   /**
    * Get the value (as an Object) from JSON, based on key.
    * If not found, or the value is ".", then return null
    */
   protected Object getFromJson(JSONObject jsonObj, String key)
   {
      if (!jsonObj.has(key))
         return null;
      Object val = jsonObj.get(key);
      if (val.toString().equals("."))
         return null;
      return val;
   }


}
