package edu.mayo.bior.pipeline;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;

import edu.mayo.bior.pipeline.InfoMetaObject.KeyValType;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryMetaData;
import edu.mayo.pipes.util.metadata.AddMetadataLines;
import edu.mayo.pipes.util.metadata.AddMetadataLines.BiorMetaControlledVocabulary;

/** Build the INFO column up from other columns, or add specific key-value pairs to INFO.
 *  This is used primarily by the TjsonToVcfPipe */
public class VcfInfoColumnBuilder {

	/** How to encode values to insert into the VCF INFO field.
	 *   - SIMPLE will try to keep all conversions as single character changes (spaces to underscores, commas to pipes, 
	 *   - URL_BASIC will convert commas, semicolons, equals
	 *   - URL_EXTENDED will convert commas, semicolons, equals, percent, colon	 */
	public enum CharacterEncodingMethod { SIMPLE, URL_BASIC, URL_EXTENDED };
	private CharacterEncodingMethod characterEncodingMethod = CharacterEncodingMethod.URL_BASIC;
	
	private Map<Character,String> characterEncodingMap = new HashMap<Character,String>();

	public VcfInfoColumnBuilder() {
		setCharacterEncodingMethod(CharacterEncodingMethod.URL_BASIC); // SIMPLE);
	}
		
	//private static final int INFO_COLUMN = 7;

	//public final String DEFAULT_DESCRIPTION = "BioR property file missing description";
    //public final String DEFAULT_TYPE = "String";
	//public final String DEFAULT_NUMBER = ".";
    
	private static final Logger sLogger = Logger.getLogger(VcfInfoColumnBuilder.class);
	
    /** During the queue-filling stage at first, we gather a list of X lines (ex: 1000) to
     *  determine what all of the key-value pairs should be when constructing the ##INFO header lines     */
    private List<History> mHistoryLinesToUseForInfoGathering = new ArrayList<History>();
    
	// Map:  BiorHeaderId -> LineKey -> LineValue
	//  ex:  "bior.dbSNP_142_GRCh37p13" -> "ShortUniqueName" -> "dbSNP_142_GRCh37p13"
    // From example ##BIOR header
    //  ##BIOR=<ID="bior.dbSNP_142_GRCh37p13",Operation="bior_overlap",DataType="JSON",ShortUniqueName="dbSNP_142_GRCh37p13",Source="dbSNP",Description="NCBI's dbSNP Variant Database",Version="142",Build="GRCh37.p13",Path="/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v1/00-All.vcf.tsv.bgz">
    protected Map<String,LinkedHashMap<String, String>> mBiorHeaderMap = null;
    
    // Map: catalogPath -> columns.tsv contents
    private Map<String,String> mBiorCatalogPathToColumnsTsvContentsMap = new HashMap<String,String>();
    
	private TreeSet<Integer> mZeroBasedColumnsToAdd = new TreeSet<Integer>(Collections.reverseOrder());
    
    /** The original column header line (ex:  "#CHROM  POS  ID...") before manipulation of the headers after the first line is processed. 
     *  This is used to refer back to the column header when manipulating data.     */
    private List<String> mOriginalColumnHeader;
    
    /** The original column metadata (info about each column) before manipulation of the headers after the first line is processed.
     *  This is used to refer back to the column header when manipulating data.     */
	private List<ColumnMetaData> mOriginalColumnMetaData;
    
	private Comparator<InfoMetaObject> mInfoMetaObjCompareById;

	/** A Map between InfoMetaObject.getId() and the full InfoMetaObject */ 
	private Map<String, InfoMetaObject> mInfoMetaMap = new HashMap<String, InfoMetaObject>();

	private JsonParser jsonParser = new JsonParser();

	/** Get the character encoding method used on values inserted into the VCF INFO field.
	 *   - SIMPLE will try to keep all conversions as single character changes (spaces to underscores, commas to pipes, 
	 *   - URL_BASIC will convert commas, semicolons, equals
	 *   - URL_EXTENDED will convert commas, semicolons, equals, percent, colon
	 *  Default is URL_BASIC	 */
	public CharacterEncodingMethod getCharacterEncodingMethod() {
		return this.characterEncodingMethod;
	}
	
	/** Get the character encoding method used on values inserted into the VCF INFO field. */	
	public void setCharacterEncodingMethod(CharacterEncodingMethod charEncodingMethod) {
		this.characterEncodingMethod = charEncodingMethod;
		
		this.characterEncodingMap = new HashMap<Character,String>();
		if( this.characterEncodingMethod.equals(CharacterEncodingMethod.URL_BASIC) 
		 || this.characterEncodingMethod.equals(CharacterEncodingMethod.URL_EXTENDED))
		{
			this.characterEncodingMap.put(',', "%2C");
			this.characterEncodingMap.put('=', "%3D");
			this.characterEncodingMap.put(';', "%3B");
		}
		
		if( this.characterEncodingMethod.equals(CharacterEncodingMethod.URL_EXTENDED) ) {
			this.characterEncodingMap.put(':', "%3A");
			this.characterEncodingMap.put('%', "%25");
			this.characterEncodingMap.put('\t',"%09");
			this.characterEncodingMap.put('\r',"%0D");
			this.characterEncodingMap.put('\n',"%0A");			
		}
	}
	
    /**
     * Create the INFO metadata based on the columns to add to the INFO column.
     * This will be done in two passes:
     *   1) Compare column header to ##BIOR header rows - if match, then add corresponding ##INFO
     *   2) If not already added, then add the ##INFO row based on the buffered line queue
     * @param lineQueue  The list of lines to pull data from
     * @param mColumnsToAddToInfo  The columns within the lines that contains the fields to add to the INFO metadata "##" headers.
     *             NOTE: These column indexes must be 0-based integers from 0 to (maxColLength - 1)
     * @return  List of ##INFO metadata strings to add to header
     */
	public List<String> createInfoMetadata(List<History> lineQueue,  TreeSet<Integer> mColumnsToAddToInfo) {
		mHistoryLinesToUseForInfoGathering = lineQueue;
		mZeroBasedColumnsToAdd = mColumnsToAddToInfo;

		HistoryMetaData meta = lineQueue.get(0).getMetaData();
		mBiorHeaderMap = getBiorHeaderIdMap(meta);
		
        mOriginalColumnHeader = Arrays.asList(meta.getColumnHeaderRow(lineQueue.get(0), "\t").split("\t",-1));
        mOriginalColumnMetaData = Collections.unmodifiableList(new ArrayList<ColumnMetaData>(meta.getColumns()));

        // Get metadata from the ##BIOR header lines if possible - this should be most reliable as it should come from the catalog's columns.tsv file
        List<InfoMetaObject> infoMetasFromBiorHeaders = getInfoMetaFromBiorHeaders(mColumnsToAddToInfo);
        
        // Get metadata from the data within the lineQueue.  This is a little less reliable, but should still give us a decent guess at the datatypes since we can look at actual examples of the data.
		List<InfoMetaObject> infoMetasFromQueue = getInfoMetaObjectsFromQueuedLines(lineQueue, mColumnsToAddToInfo);

		// Lastly, get metadata from the column headers.  This is least reliable since all we have to go off of is the column header name.
		// The number/count defaults to ".", and the type defaults to "String"
		List<InfoMetaObject> infoMetasFromColumnHeadersGeneric = getInfoMetaObjectsFromColumnHeadersGeneric(mColumnsToAddToInfo);
		
		
		
		// Create a merged list of InfoMetaObjects from the ones from ##BIOR headers and from the lines in the queue
		// The ones from ##BIOR take precedence.  The ones based on keys in the data rows comes next
		List<InfoMetaObject> infoMetasTemp = mergeInfoMetas(infoMetasFromBiorHeaders, infoMetasFromQueue);
		
		// Now merge those with the metadata returned directly from the header rows
		List<InfoMetaObject> infoMetas = mergeInfoMetas(infoMetasTemp, infoMetasFromColumnHeadersGeneric);
		
		
		
		// Add the InfoMetaObjects to the class's Map (map id to InfoMetaObject)
		// Then we can re-use these when building the data INFO column  (we can quickly lookup by the Map's key)
		for(InfoMetaObject infoMeta : infoMetas)
			mInfoMetaMap.put(infoMeta.getId(), infoMeta);
		
		// Convert InfoMetaObjects to ##INFO meta lines
		return infoMetasToStrings(infoMetas);
	}

	private List<InfoMetaObject> getInfoMetaObjectsFromColumnHeadersGeneric(TreeSet<Integer> columnsToAddToInfo) {
		List<InfoMetaObject> infoMetaObjects = new ArrayList<InfoMetaObject>();
		History line = mHistoryLinesToUseForInfoGathering.get(0);
		for(int colToAdd : columnsToAddToInfo) {
			ColumnMetaData colMeta = line.getMetaData().getColumns().get(colToAdd);

			// Skip the column if it is a JSON column, since we don't want a separate ##INFO line for whole JSON columns, only individual keys
			if( isBiorHeaderTypeJson(colMeta.getColumnName())  ||  isAnyJsonInColumn(colToAdd) )
				continue;
			
			String count = colMeta.getCount();
			if( count == null  ||  count.trim().length() == 0 )
				count = ".";
			
			String desc = colMeta.getDescription();
			if( desc == null || desc.trim().length() == 0 )
				desc = "";
			
			infoMetaObjects.add( new InfoMetaObject(
					"",
					new ArrayList<String>(), 
					colMeta.getColumnName(),
					getTypeFromColumnMetaData(line, colToAdd),
					count,
					desc ) );
		}
		return infoMetaObjects;
	}

	private boolean isAnyJsonInColumn(int colZeroBased) {
		for(History line : mHistoryLinesToUseForInfoGathering) {
			if( isJsonObject(line.get(colZeroBased)) )
				return true;
		}
		return false;
	}

	private List<InfoMetaObject> mergeInfoMetas(List<InfoMetaObject> infoMetasPrimary, List<InfoMetaObject> infoMetasSecondary) {
		List<InfoMetaObject> infoMetas = new ArrayList<InfoMetaObject>();
		
		// Add all the primaries to the list
		for(InfoMetaObject infoMetaPrimary : infoMetasPrimary) {
			infoMetas.add(infoMetaPrimary);
		}
		
		// Loop thru the secondaries.  If any are not already in the list, then add them
		for(InfoMetaObject infoMetaSecondary : infoMetasSecondary) {
			boolean isInPrimaryList = false;
			for(InfoMetaObject infoMetaPrimary : infoMetasPrimary) {
				if( infoMetaPrimary.getId().equals(infoMetaSecondary.getId()) ) {
					isInPrimaryList = true;
				}
			}
			if( ! isInPrimaryList )
				infoMetas.add(infoMetaSecondary);
		}
		
		return infoMetas;
	}

	private List<String> infoMetasToStrings(List<InfoMetaObject> infoMetas) {
		// Sort the InfoMetaObjects by ID so that when we convert them to strings the will be added to the header in the same order each time
		mInfoMetaObjCompareById = getInfoMetaObjectComparator();
		Collections.sort(infoMetas, mInfoMetaObjCompareById);
		
		List<String> infoMetadataHeaders = new ArrayList<String>();
		for(InfoMetaObject infoMetaObj : infoMetas) {
			// Have to update the description since we may be able to pull more info from the ##BIOR lines and the columns.tsv files
			updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, mBiorHeaderMap);
			infoMetadataHeaders.add(infoMetaObj.toString());
		}
		return infoMetadataHeaders;
	}

	/** Looking at each column header to add to the INFO column, find the corresponding
	 *  entry in the ##BIOR headers for more info to construct the InfoMetaObject with.
	 *  PRE:  These class fields must be set:
	 *    - mOriginalColumnHeader  (List of strings of the column headers before modification)
	 *    - mBiorHeaderMap  (HashMap of different ##BIOR lines by ID, with the keys and values being the nested LinkedHashMap) */
	private List<InfoMetaObject> getInfoMetaFromBiorHeaders(TreeSet<Integer> columnsToAddToInfo)
	{
		List<InfoMetaObject>  metaFromBiorHeaders = new ArrayList<InfoMetaObject>();
		for(Integer colIdx : columnsToAddToInfo) {
			String colHeaderStr = mOriginalColumnHeader.get(colIdx);
			// Only add a new InfoMetaObject if the colHeaderStr exists in the ##BIOR lines, AND it is NOT a JSON type
			if( mBiorHeaderMap.containsKey(colHeaderStr)  &&  ! isBiorHeaderTypeJson(colHeaderStr)) {
				InfoMetaObject infoMetaObj = getInfoMetaFromBiorHeaderLine(colHeaderStr);
				
				// Updates from the columns.tsv
				updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, mBiorHeaderMap);
				
				metaFromBiorHeaders.add(infoMetaObj);
			}
		}
		return metaFromBiorHeaders;
	}


	private InfoMetaObject getInfoMetaFromBiorHeaderLine(String colHeaderStr) {
		Map<String,String> biorMetaFields = mBiorHeaderMap.get(colHeaderStr);
		
		KeyValType type = columnsTsvTypeToKeyValType(biorMetaFields.get(BiorMetaControlledVocabulary.DATATYPE.toString()));
		
		
		String numOccurrences = biorMetaFields.get(BiorMetaControlledVocabulary.NUMBER.toString());
		if( numOccurrences == null  ||  numOccurrences.trim().length() == 0 )
			numOccurrences = ".";

		// Get the field description from "FieldDescription"
		// (but not from "Description" as that is the overall catalog/datasource description)
		String description = biorMetaFields.get(BiorMetaControlledVocabulary.FIELDDESCRIPTION.toString());
		if( description == null  ||  description.trim().length() == 0 )
			description = "";

		return new InfoMetaObject(null, new ArrayList<String>(), colHeaderStr, type, numOccurrences, description);
	}

	private boolean isBiorHeaderTypeJson(String key) {
		Map<String,String> biorMetaLineFieldMap = mBiorHeaderMap.get(key);
		// If null, then there is no ##BIOR line for this key, so we can't know if it's JSON or not, so return false
		if( biorMetaLineFieldMap == null )
			return false;
		
		String type = biorMetaLineFieldMap.get(BiorMetaControlledVocabulary.DATATYPE.toString());
		// The "Type" field may not be in this line either, so check it carefully
		return "JSON".equalsIgnoreCase(type);
	}
	

	private List<InfoMetaObject> getInfoMetaObjectsFromQueuedLines(List<History> lineQueue, TreeSet<Integer> columnsToAddToInfo) {
		List<InfoMetaObject> infoMetaObjList = new ArrayList<InfoMetaObject>();
		Set<String> idSet = new HashSet<String>();
		// Loop thru all lines in Queue
		for(History history : lineQueue) {
			// Skip any blank lines
			if( isBlankLine(history) )
				continue;
			
			// Loop thru each column to add, and add all data
			for(Integer colIdxToAdd : columnsToAddToInfo) {
				List<InfoMetaObject> newObjsToAdd = getColumnAsInfoMetaObjects(history, colIdxToAdd);
				// Only add each one if it is NOT a repeat
				for(InfoMetaObject obj : newObjsToAdd) {
					if( ! idSet.contains(obj.getId()) ) {
						infoMetaObjList.add(obj);
						idSet.add(obj.getId());
					}
				}
			}
		}
		return infoMetaObjList;
	}

	private boolean isBlankLine(History history) {
		return  history == null || history.size() == 0 || (history.size() == 1 && history.get(0).length() == 0);
	}
	
	
	/** Get ##INFO lines for header metadata from the INFO field within the target JSON column that is used to build the VCF structure
	 *  Ex:  {..., "INFO":{"MAF":0.3,...}}
	 *  Ex:  {..., "INFO":"MAF=0.3;AF=0.23"}  */
	public List<String> createInfoMetadataFromInfoInTargetJsonCol(List<History> lineQueue, int jsonColumnToBuildVcfStructure) {
		List<InfoMetaObject> infoMetaObjs = new ArrayList<InfoMetaObject>();

		for(History line : lineQueue) {
			// If the line is blank, then skip it
			if( isBlankLine(line) )
				continue;
			
			// If target column is not JSON, then skip it
			String jsonCol = line.get(jsonColumnToBuildVcfStructure);
			if( ! isJsonObject(jsonCol) )
				continue;
			
			JsonObject jsonObj = stringToJsonObject(jsonCol);
			JsonElement infoElem = jsonObj.get("INFO");
			if( infoElem == null )
				continue;
			
			// If the INFO field is an object, then extract the InfoMetaData from it
			// Ex: "INFO":{"MAF":0.3,"AF":0.25}
			if( infoElem.isJsonObject() ) {
				infoMetaObjs.addAll(jsonElementToInfoMetaObjects("", "", infoElem));
			} else if( infoElem.isJsonPrimitive() ) {
				infoMetaObjs.addAll(flatInfoToInfoMetaObjects(infoElem.getAsString()));
			}
		}

		List<String> infoMetaObjStrs = new ArrayList<String>();
		Set<String> infoKeys = new HashSet<String>();
		for(InfoMetaObject infoMetaObj : infoMetaObjs) {
			if( ! infoKeys.contains(infoMetaObj.getId()) ) {
				infoMetaObjStrs.add( infoMetaObj.toString() );
				infoKeys.add(infoMetaObj.getId());
			}
		}

		return infoMetaObjStrs;
	}

	/** Take a flat INFO field and convert to List<InfoMetaObject>
	 *  Ex: "INFO":"MAF=0.3;AF=0.25;isDbsnp"
	 */
	private List<InfoMetaObject> flatInfoToInfoMetaObjects(String infoFlat) {
		List<InfoMetaObject> infoMetaObjs = new ArrayList<InfoMetaObject>();
		// Split the values
		String[] keyVals = infoFlat.split(";");
		for(String keyVal : keyVals) {
			if( keyVal.length() == 0  ||  keyVal.equals("."))
				continue;
			String[] keyOrVal = keyVal.split("=");
			String key = keyOrVal[0];
			String[] vals = (keyOrVal.length == 1  ? "" : keyOrVal[1]).split(",");
			KeyValType type = getType(vals[0]);
			String count = KeyValType.Flag.equals(type) ? "0" : ".";
			infoMetaObjs.add(new InfoMetaObject(key, Arrays.asList(vals), "", type, count, ""));
		}
		return infoMetaObjs;
	}

	/**	Look at parent column, find it's ##BIOR line if applicable.
	 *     If ##BIOR line not found, or columns.tsv not found, or key within columns.tsv not found, then do nothing - (no updates, description should be "")
	 *     If ##BIOR line found AND has catalog path, then lookup key in columns.tsv:
	 *     		If key is found there, then update its type, number, description from columns.tsv row
	 * @param infoMetaObj  The InfoMetaObject to update
	 * @param biorHeaderMap  HashMap of key-value pairs from the ##BIOR lines
	 * @throws IOException 
	 */
	protected void updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(InfoMetaObject infoMetaObj,  Map<String, LinkedHashMap<String, String>> biorHeaderMap) {
		LinkedHashMap<String,String> biorMetaMapForKey = biorHeaderMap.get(infoMetaObj.parentColumnName);
		if( biorMetaMapForKey == null ) 
			return;
		
		// If we don't already have the contents of the columns.tsv file for the catalog, then fetch it
		String catalogPath = biorMetaMapForKey.get(BiorMetaControlledVocabulary.PATH.toString());
		addColumnsTsvToMap(catalogPath);
		
		// Lookup the key in the columns.tsv file and if it matches, then update the type, numberOfOccurrences, and Description
		String columnsTsvContents = mBiorCatalogPathToColumnsTsvContentsMap.get(catalogPath);
		if( columnsTsvContents == null ) 
			return;

		updateInfoMetaObjFromColumnsTsvContents(infoMetaObj, biorMetaMapForKey, columnsTsvContents);

	}

	private void updateInfoMetaObjFromColumnsTsvContents(InfoMetaObject infoMetaObj, LinkedHashMap<String, String> biorMetaMapForKey, String columnsTsvContents) {
		String[] rows = columnsTsvContents.split("\n");
		
		// If the ##BIOR line that matches the column header contains the "Field" key, this means the field was drilled
		//   so we should have the name that matches the name in the columns.tsv file
		// Else if the key is null (the column is just a value with the key being the header), use the parentColumnName to match the value in the columns.tsv
		String dataKey = infoMetaObj.jsonKey;
		if( biorMetaMapForKey.containsKey(BiorMetaControlledVocabulary.FIELD.toString()) )
			dataKey = biorMetaMapForKey.get(BiorMetaControlledVocabulary.FIELD.toString());
		if( dataKey == null  ||  dataKey.length() == 0 )
			dataKey = infoMetaObj.parentColumnName;
		
		for(String row : rows) {
			String[] cols = row.split("\t");
			// TODO: MIKE:  Verify that each column is present before assigning it.  Assign defaults????
			if( cols.length > 0 ) {
				String keyInColumnsCsv = cols[0].trim();
				if( ! keyInColumnsCsv.startsWith("#")  &&  keyInColumnsCsv.equalsIgnoreCase(dataKey) ) {
					// Columns:  #ColumnName     Type    Count   Description	HumanReadableName
					infoMetaObj.type = columnsTsvTypeToKeyValType(cols[1].trim());
					infoMetaObj.numOccurrences = cols[2].trim();
					infoMetaObj.description = cols[3].trim();
				}
			}
		}
	}

	/** Find the columns.tsv for the given catalog, load it, and save it to the mBiorCatalogPathToColumnsTsvContentsMap if possible */
	private void addColumnsTsvToMap(String catalogPath) {
		if( catalogPath == null ) 
			return;
		
		if( mBiorCatalogPathToColumnsTsvContentsMap.get(catalogPath) == null ) {
			// Try to get columns.tsv from catalog path
			File columnsTsvFile = new File(catalogPath.replace(".tsv.bgz", "").replace(".tsv.gz", "") + ".columns.tsv");
			if( ! columnsTsvFile.exists() )
				return;
			
			try {
				String columnsTsvContents = FileUtils.readFileToString(columnsTsvFile);
				mBiorCatalogPathToColumnsTsvContentsMap.put(catalogPath, columnsTsvContents);
			}catch(Exception e) {
				sLogger.warn("Could not read from columns.tsv file: " + columnsTsvFile.toString() + "  for catalog: " + catalogPath);
				return;
			}
		}
	}

	/** Convert a Type from the columns.tsv file (Possible values: JSON, String, Integer, Float, Boolean) to a KeyValType */
	private KeyValType columnsTsvTypeToKeyValType(String columnsTsvType) {
		if( columnsTsvType == null  ||  columnsTsvType.trim().length() == 0 )
			return KeyValType.String;
		if( columnsTsvType.equalsIgnoreCase("Integer") ) 
			return KeyValType.Integer;
		else if( columnsTsvType.equalsIgnoreCase("Float") ) 
			return KeyValType.Float;
		else if( columnsTsvType.equalsIgnoreCase("Boolean") ) 
			return KeyValType.Flag;
		else
			return KeyValType.String;
	}

	/** Add all Strings/JSON/JSON-Arrays from the specified list of columns into the INFO column.
	 *  As each key-value pair is added, a new INFO metadata row is created and also added to the metadata if it is not already there  
	 * @param history  The current data line from which to grab columns to collapsed 
	 * @param mColumnsToAddToInfo  The list of columns to collapse into the INFO column (and remove)
     *             NOTE: These column indexes must be 0-based integers from 0 to (maxColLength - 1)
	 * @return  The INFO column that will be merged with the existing History object's INFO column
	 * 
	 */
	public String createInfoData(History history,	TreeSet<Integer> mColumnsToAddToInfo) {
		// Loop thru each column to add, and add all data
		List<InfoMetaObject> infoMetaObjs = new ArrayList<InfoMetaObject>();
		for(int colIdxToAdd : mColumnsToAddToInfo) {
			infoMetaObjs.addAll( getColumnAsInfoMetaObjects(history, colIdxToAdd) );
		}
		Collections.sort(infoMetaObjs, mInfoMetaObjCompareById);
		String newInfoMetaObjs = getInfoMetaObjsAsString(infoMetaObjs);
		return newInfoMetaObjs;
	}




	/** Sort info InfoMetaObjects by id */
	private Comparator<InfoMetaObject> getInfoMetaObjectComparator() {
		return new Comparator<InfoMetaObject>() {
			public int compare(InfoMetaObject info1, InfoMetaObject info2) {
				return info1.getId().compareToIgnoreCase(info2.getId());
			}
		};
	}



	/** Add the ##INFO line just above the column header line */
	public void addInfoLineToHeader(History history, String infoMetaLine) {
		List<String> metaHeaders = history.getMetaData().getOriginalHeader();
		int rowToInsertBefore = getIdxToInsertInfo(metaHeaders, infoMetaLine);
		metaHeaders.add(rowToInsertBefore, infoMetaLine);
	}


   
   /** Get the index within the header rows where we should insert this infoLine.
    *  It should go before the column header line, and should be inserted alphabetically amongst other ##INFO columns */
   private int getIdxToInsertInfo(List<String> metaHeaders,  String infoMetaLine) {
	   boolean isAnInfoLineFound = false;
	   // Start from the bottom of the header list and work up.  
	   for(int i=metaHeaders.size()-1; i >= 0; i--) {
		   String meta = metaHeaders.get(i);
		   boolean isInfoRow = meta.startsWith("##INFO");
		   if( isInfoRow ) {
			   if( infoMetaLine.compareToIgnoreCase(meta) > 0 )
				   return i+1;
			   else // We found a ##INFO, but there may be others to compare against
				   isAnInfoLineFound = true;
		   } else if( isAnInfoLineFound ) {  // This would be the first row BEFORE the found ##INFO, so insert after it
			   return i+1;
		   }   
	   }
	   // No column header row or ##INFO line found that was after infoMetaLine alphabetically, so insert before the column header line
	   // NOTE: Don't go below 0!  (if metaHeaders.size() == 0)
	   // NOTE: If a ##INFO line was the top-most line, then return 0;
	   // NOTE: Also return 0 if metaHeaders.size() == 0
	   if( isAnInfoLineFound  ||  metaHeaders.size() == 0 )
		   return 0;
	   else
		   return metaHeaders.size()-1;
   }

/**
     * generate a key-value hash for the ##BIOR lines in the header.
     * (we have a similar hash for column number, but this one is for those lines that specify that some column should be treated as an array)
     * MAP: ID->LineKey->Value
     * Example:
     * 		"bior.geneName" ->	[
     * 							"Type" -> "Integer"
     *      					"ID"   -> "bior.geneName"
     *     						"Description" -> "Gene name"
     *     						]
     *     
     * Example ##BIOR header lines:
 	   		##BIOR=<ID="bior.dbSNP_142_GRCh37p13",Operation="bior_lookup",DataType="JSON",ShortUniqueName="dbSNP_142_GRCh37p13",Source="dbSNP",Description="NCBI's dbSNP Variant Database",Version="142",Build="GRCh37.p13",Path="/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v2/00-All.vcf.tsv.bgz">
			##BIOR=<ID="bior.dbSNP_142_GRCh37p13.ID",Operation="bior_drill",Field="ID",DataType="String",Number=".",FieldDescription="Semi-colon separated list of unique identifiers.  If this is a dbSNP variant, the rs number(s) should be used.  (VCF field)",ShortUniqueName="dbSNP_142_GRCh37p13",Source="dbSNP",Description="NCBI's dbSNP Variant Database",Version="142",Build="GRCh37.p13",Path="/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v2/00-All.vcf.tsv.bgz">
			##BIOR=<ID="bior.dbSNP_142_GRCh37p13._landmark",Operation="bior_drill",Field="_landmark",DataType="String",Number="1",FieldDescription="Provides a context for the genomic coordinates _minBP and _maxBP.  Most often this is the chromosome where the feature appears, but could be a known genetic marker, gene, or other item. (BioR field)",ShortUniqueName="dbSNP_142_GRCh37p13",Source="dbSNP",Description="NCBI's dbSNP Variant Database",Version="142",Build="GRCh37.p13",Path="/research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v2/00-All.vcf.tsv.bgz">
	   		##BIOR=<ID="bior.#UNKNOWN_4._landmark",Operation="bior_drill",DataType="String",ShortUniqueName="",Path="">
     */
    protected Map<String, LinkedHashMap<String, String>> getBiorHeaderIdMap(HistoryMetaData historyMetaData){
    	Map<String, LinkedHashMap<String, String>> biorHeaderMap = new HashMap<String, LinkedHashMap<String,String>>();
        AddMetadataLines amdl = new AddMetadataLines();
        List<String> headerLines = historyMetaData.getOriginalHeader();
        for(String headerLine : headerLines){
        	LinkedHashMap<String, String> keyVal = amdl.parseHeaderLine(headerLine);
        	String id = keyVal.get(AddMetadataLines.BiorMetaControlledVocabulary.ID.toString());
        	biorHeaderMap.put(id, keyVal);
        }
        return biorHeaderMap;
    }



    /** Convert a list of InfoMetaObject objects to a string to insert into the INFO column.  If the list is empty, return "" */
	protected String getInfoMetaObjsAsString(List<InfoMetaObject> infoMetaObjs) {
		StringBuilder keyValsStr = new StringBuilder();
		for(int i=0; i < infoMetaObjs.size(); i++) {
			InfoMetaObject infoMetaObj = infoMetaObjs.get(i);
			if( infoMetaObj.type.equals(KeyValType.Flag) ) {
				if( "true".equalsIgnoreCase(infoMetaObj.val.get(0)) )
					keyValsStr.append(infoMetaObj.getId()).append(";");
			// Else only add the key-value-pair if there are some values (don't add empty arrays for instance)
			} else if( infoMetaObj.val.size() > 0 ) {
				keyValsStr.append(infoMetaObj.getId()).append("=");
				for( int j=0; j < infoMetaObj.val.size(); j++ ) {
					if( j > 0 )  // Separate VALUES
						keyValsStr.append(",");
					keyValsStr.append(infoMetaObj.val.get(j));
				}
				// Separate KeyVal pairs
				keyValsStr.append(";");
			}
		}
		// If the string ends with ";", then remove it
		if( keyValsStr.length() > 0  &&  keyValsStr.charAt(keyValsStr.length()-1) == ';' )
			keyValsStr.deleteCharAt(keyValsStr.length()-1);
		return keyValsStr.toString();
	}

	/** Split the column (0-based integer from 0 to n) into InfoMetaObjects<br>
	 *  NOTE: createInfoMetadata() MUST be called before this to set the original column metadata */
	protected List<InfoMetaObject> getColumnAsInfoMetaObjects(History history, Integer columnNumZeroBasedNonNeg) {
		String colData = history.get(columnNumZeroBasedNonNeg);
		List<InfoMetaObject> infoObjList = new ArrayList<InfoMetaObject>();
		
		// Set the parent column name - MAKE SURE TO USE THE ORIGINAL COLUMN NAMES, NOT THE ONES AFTER MANIPULATING THE HEADER!!!
		String columnHeaderName = getOriginalColumnMetaData(history).get(columnNumZeroBasedNonNeg).getColumnName();

		if( colData == null || colData.length() == 0 || colData.equals(".") ) {
			// Do nothing
		}else if( isJsonObject(colData) ) {
			infoObjList.addAll( jsonObjectToInfoMetaObjects("", columnHeaderName, stringToJsonObject(colData)) );
		} else if( isJsonArray(colData) ) {
			infoObjList.add(    jsonArrayToInfoMetaObject("", columnHeaderName, stringToJsonArray(colData)) );
		} else  { // Could be a string, integer, float, flag, etc - need to check the header for more info, if available.
			infoObjList.add(    stringToInfoMetaObject(history, columnNumZeroBasedNonNeg) );
		}

		return infoObjList;
	}

	private JsonObject stringToJsonObject(String s) {
		JsonElement infoElem = this.jsonParser.parse(s);
		JsonObject jsonObj = infoElem.getAsJsonObject();
		return jsonObj;
	}
	
	private JsonArray stringToJsonArray(String s) {
		JsonElement infoElem = this.jsonParser.parse(s);
		JsonArray jsonArray = infoElem.getAsJsonArray();
		return jsonArray;
	}
	
	

	protected boolean isJsonObject(String colData) {
		boolean isJsonObj = false;
		if( colData.startsWith("{") && colData.endsWith("}") ) {
			try {
				stringToJsonObject(colData);
				isJsonObj = true;
			} catch(Exception e) {
				isJsonObj = false;
			}
		}
		return isJsonObj;
	}

	protected boolean isJsonArray(String colData) {
		boolean isJsonArray = false;
		if( colData.startsWith("[") && colData.endsWith("]") ) {
			try {
				stringToJsonArray(colData);
				isJsonArray = true;
			} catch(Exception e) {
				isJsonArray = false;
			}
		}
		return isJsonArray;
	}


	/** If the info value is a JSON object, then break it up into standard VCF format.
	    Ex:  INFO:{"AC":3,"MAF":0.05,"NA":[2,3,4],"Gene":"BRCA1","isInDbsnp":true,"isIn1000Genomes":false}
	           ==>  AC=3;MAC=0.05;NA=2,3,4;Gene=BRCA1;isInDbsnp
	    (where flags (true/false) will have the flag name if true, or not include the flag if false
	    NOTE: This can be used recursively when there is a JSON object nested within another JSON object!   
	 * @param columnHeaderName */
	private List<InfoMetaObject> jsonObjectToInfoMetaObjects(String keyParent, String columnHeaderName, JsonObject jsonObj) {
		Set<Entry<String,JsonElement>> entrySet = jsonObj.entrySet();
		Entry<String,JsonElement>[] entries = entrySet.toArray(new Entry[0]);
		List<InfoMetaObject> infoObjList = new ArrayList<InfoMetaObject>();
		
		// Add each key-value pair found in the JSON
		for(int i=0; i < entries.length; i++) {
			JsonElement elem = (JsonElement)(entries[i].getValue());
			String parentPrefix = (keyParent == null || keyParent.length() == 0)  ?  ""  :  keyParent + ".";
			String key = parentPrefix + (String)(entries[i].getKey());
			infoObjList.addAll(jsonElementToInfoMetaObjects(key, columnHeaderName, elem));
		}
		
		return infoObjList;
	}
	
	
	/** Flatten a JSON object to a string.
	 *  Ex: {"MAF":0.35,"AF":[0.2,0.1,0.05],"AF2":{"EUR":0.2,"AFR":0.11},"isDbsnp":false,"isUcsc":true,"AString":"aValue","Dud":"."}
	 *  To: MAF=0.35;AF=0.2,0.1,0.05;AF2.EUR=0.2;AF2.AFR=0.11;isUcsc;AString=aValue
	 * @param parentKey The JSON parent key (since we may be recursing for example, this could be "AF2"
	 *  				in the example above, where we have to dig deeper into a nested JSON object)
	 * @param jsonObj JSON object (possibly nested) to flatten
	 * @return  String equivalent of the flattened JSON object
	 */
	public String jsonObjectToInfoDataString(String parentKey, JsonObject jsonObj) {
		StringBuilder info = new StringBuilder();
		String prefix = (parentKey == null || parentKey.length() == 0)  ?  ""  : (parentKey + ".");
		Iterator<Entry<String,JsonElement>> iterator = jsonObj.entrySet().iterator();
		while(iterator.hasNext()) {
			Entry<String,JsonElement> keyVal = iterator.next();
			String key = prefix + keyVal.getKey();
			String separator = info.length() > 0  ?  ";"  :  "";
			JsonElement val = keyVal.getValue();
			// Recurse thru objects
			if( val.isJsonObject() )
				info.append(separator + jsonObjectToInfoDataString(key, val.getAsJsonObject()));
			
			// Add all array items
			else if( val.isJsonArray() ) {
				info.append(separator + key + "=");
				for(int i=0; i < val.getAsJsonArray().size(); i++) {
					if( i > 0 )
						info.append(",");
					info.append(val.getAsJsonArray().get(i));
				}
			}
			
			// Add primitive - booleans, ints, strings 
			else if( val.isJsonPrimitive() ) {
				if( val.getAsJsonPrimitive().isBoolean() ) {
					if( val.getAsJsonPrimitive().getAsBoolean() )
						info.append(separator + key);
				}
				// Else is an int, float, string
				else {
					String valNoQuotes = removeQuotes(val.toString());
					if( ! valNoQuotes.equals(".") )
						info.append(separator + key + "=" + valNoQuotes);
				}
			}
		}
		
		return info.toString();
	}


	/** Given a JsonElement (usually when embedded within a JSON object), 
	 *  figure out what type of element it is and return the key-value-pairs for that type.	 */
	private List<InfoMetaObject> jsonElementToInfoMetaObjects(String key, String columnHeaderName, JsonElement elem) {
		// If it's an object, then it is nested, so need to pull it apart
		if( elem.isJsonObject() ) {
			return jsonObjectToInfoMetaObjects(key, columnHeaderName, (JsonObject)elem);
		// If it's an array, add all elements (comma-separated)
		} else if( elem.isJsonArray() ) {
			return Arrays.asList(jsonArrayToInfoMetaObject(key, columnHeaderName, (JsonArray)elem));
		// else is a primitive: number, string, or boolean
		} else {
			InfoMetaObject infoMeta = jsonPrimitiveToInfoMetaObject(key, columnHeaderName, (JsonPrimitive)elem);
			return infoMeta == null  ?  new ArrayList<InfoMetaObject>()  :  Arrays.asList(infoMeta);
		}
	}
	
	/** Get a JSON array as a string to insert into the INFO column
	 * @param columnHeaderName 
	 * @param jsonArray JsonArray to convert to key with a set of values
	 */
	private InfoMetaObject jsonArrayToInfoMetaObject(String jsonKey, String columnHeaderName, JsonArray jsonArray) {
		String id = InfoMetaObject.getId(jsonKey, columnHeaderName);
		
		InfoMetaObject infoMetaObj = mInfoMetaMap.get(id);
		
		// If the infoMetaObj is not in already in the map, then create a new one and add it to the map
		if( infoMetaObj == null ) {
			infoMetaObj = new InfoMetaObject();
			infoMetaObj.jsonKey = jsonKey;
			infoMetaObj.parentColumnName = columnHeaderName;
			infoMetaObj.type = getTypeArray(jsonArray);
			infoMetaObj.numOccurrences = ".";
			
			// Add to the Map
			mInfoMetaMap.put(id, infoMetaObj);
		}
		
		// Still need to add values in case we are processing the data lines to put into the INFO column
		// NOTE: Make sure to CLEAR out the list as we don't want values from a previous row 
		infoMetaObj.val = new ArrayList<String>();
		for(int j=0; j < jsonArray.size(); j++) {
			String val = replaceBadCharsAndMetaDelimsInValue("", jsonArray.get(j).toString());
			infoMetaObj.val.add(val);
		}
		
		return infoMetaObj;
	}
	
	protected KeyValType getTypeArray(JsonArray jsonArray) {
		List<KeyValType> typeList = new ArrayList<KeyValType>();
		
		for( int i=0; i < jsonArray.size(); i++ ) {
			typeList.add(getType(jsonArray.get(i)));
		}
		
		return pickMostGeneralType(typeList);
	}

	protected KeyValType getType(JsonElement jsonElem) {
		if( ! jsonElem.isJsonPrimitive() )
			return KeyValType.String;
		
		JsonPrimitive prim = (JsonPrimitive)jsonElem;
			
		// Flag?
		if( prim.isBoolean() )
			return KeyValType.Flag;
			
		if( prim.isNumber() ) {
			// Integer?  May have to remove the ".0" at end to check if integer
			String primNoQuotes = prim.toString().replaceAll("\"", "");
			if( isInteger(primNoQuotes)  ||  (primNoQuotes.endsWith(".0")  &&  isInteger(primNoQuotes.replace(".0",""))) )
				return KeyValType.Integer;
			
			// Double/Float?
			if( isDouble(primNoQuotes) )
				return KeyValType.Float;
		}
		
		// String?
		return KeyValType.String;
	}
	
	/** Pass in a value (typically from an INFO column).
	 *  If null or blank, then return Flag   (but NOT "false" or "true", since those will appear as strings within the INFO column, such as "isDbsnp=true" instead of the flag "isDbsnp")
	 *  Else if is int, then return Integer 
	 *  Else if is double, then return Float
	 *  Else return String 	 */
	protected KeyValType getType(String val) {
		if( val == null || val.trim().length() == 0 )
			return KeyValType.Flag;
		else if( isInteger(val) )
			return KeyValType.Integer;
		else if( isDouble(val) )
			return KeyValType.Float;
		else
			return KeyValType.String;
	}

	
	private KeyValType pickMostGeneralType(List<KeyValType> typeList) {
		if( typeList.contains(KeyValType.String) )
			return KeyValType.String;
		if( typeList.contains(KeyValType.Float) )
			return KeyValType.Float;
		if( typeList.contains(KeyValType.Integer) )
			return KeyValType.Integer;
		if( typeList.contains(KeyValType.Flag) )
			return KeyValType.Flag;
		// Not sure, or empty list, so return String
		return KeyValType.String;
	}


	/** Get a JSON primitive (Boolean, Number, String) as a string to insert into the INFO column
	 * @param parentJsonKey The parent key within the JSON data (empty if this is the highest level in the JSON object)
	 * @param columnHeaderName  The column header name
	 * @param jsonPrimitive  The JSON primitive to add to a new InfoMetaObject
	 * @return  The InfoMetaObject (or null if there are no values or the values are blank/emptyString)
	 */
	private InfoMetaObject jsonPrimitiveToInfoMetaObject(String parentJsonKey, String columnHeaderName, JsonPrimitive jsonPrimitive) {
		String id = InfoMetaObject.getId(parentJsonKey, columnHeaderName);
		
		InfoMetaObject infoMetaObj = mInfoMetaMap.get(id);
				
		// If the infoMetaObj is not in already in the map, then create a new one and add it to the map
		if( infoMetaObj == null ) {
			infoMetaObj = new InfoMetaObject();
			infoMetaObj.jsonKey = parentJsonKey;
			infoMetaObj.parentColumnName = columnHeaderName;
			infoMetaObj.type = getType(jsonPrimitive);
			infoMetaObj.numOccurrences = jsonPrimitive.isBoolean() ? "0" : "1";
			
			// Add to the Map
			mInfoMetaMap.put(id, infoMetaObj);
		}

		// Add values - still need to do this in case we are processing the data lines to put into the INFO column
		// If it's a primitive, only add the key, but only if it's true.  If false, don't add key
		// NOTE: Make sure to CLEAR out the list as we don't want values from a previous row
		infoMetaObj.val = new ArrayList<String>();
		if( jsonPrimitive.isBoolean() ) {
			infoMetaObj.val.add(jsonPrimitive.toString());
		} else {// String or number, so add key=val
			String val = replaceBadCharsAndMetaDelimsInValue(parentJsonKey, removeDecimalPointIfInteger(jsonPrimitive));
			if( val == null || val.length() == 0 )
				return null;
			infoMetaObj.val.add(val);
		}

		return infoMetaObj;
	}



	private boolean isInteger(String val) {
		try {
			Integer.parseInt(val);
			return true;
		} catch(Exception e) {
			return false;
		}
	}

	private boolean isDouble(String val) {
		try {
			Double.parseDouble(val);
			return true;
		} catch(Exception e) {
			return false;
		}
	}


	private String removeDecimalPointIfInteger(JsonPrimitive jsonPrimitive) {
    	String val = jsonPrimitive.getAsString();
		if( jsonPrimitive.isNumber()  &&  val.endsWith(".0") )
			val = val.substring(0, val.length() - 2);
		return val;
	}


	/**
     * Given a key and value (from column data), return the equivalent INFO key-value-pair     
	 * @param history */
    protected InfoMetaObject stringToInfoMetaObject(History history, Integer columnNum) {
    	String jsonKey = "";
		// Need to reference the ***ORIGINAL*** column headers since the headers have already been manipulated by this point
		String columnHeaderName = getOriginalColumnMetaData(history).get(columnNum).getColumnName();
    	
		String id = InfoMetaObject.getId(jsonKey, columnHeaderName);
		
		InfoMetaObject infoMetaObj = mInfoMetaMap.get(id);
		
		// If the infoMetaObj is not in already in the map, then create a new one and add it to the map
		if( infoMetaObj == null ) {
			infoMetaObj = new InfoMetaObject();
			// The key should only be filled in when processing JSON within a column
			infoMetaObj.jsonKey = jsonKey;
			infoMetaObj.parentColumnName = columnHeaderName;
			infoMetaObj.type = getTypeFromColumnMetaData(history, columnNum);
			infoMetaObj.numOccurrences = getNumOccurrencesFromColumnMetaData(history, columnNum);
			infoMetaObj.description = getDescFromColumnMetaData(history, columnNum);
			
			// Add to the Map
			mInfoMetaMap.put(id, infoMetaObj);
		}
		
		// Add values still, in case we are processing the data lines to put into the INFO column
		// NOTE: Make sure to CLEAR out the list as we don't want values from a previous row 
		infoMetaObj.val = new ArrayList<String>();
		infoMetaObj.val.add(replaceBadCharsAndMetaDelimsInValue(columnHeaderName, history.get(columnNum)));

		return infoMetaObj;
    }
    
    
    
    private List<ColumnMetaData> getOriginalColumnMetaData(History history) {
    	if( mOriginalColumnMetaData == null )
    		mOriginalColumnMetaData = history.getMetaData().getColumns();
    	return mOriginalColumnMetaData;
	}

	private String getNumOccurrencesFromColumnMetaData(History history, Integer columnNum) {
		String numOccur = ".";
		if( history.getMetaData() != null  &&  history.getMetaData().getColumns() != null  && history.getMetaData().getColumns().size() > columnNum ) {
			numOccur = history.getMetaData().getColumns().get(columnNum).getCount();
			if( numOccur == null  || numOccur.length() == 0 )
				numOccur = ".";
		}
		return numOccur;

	}

	private String getDescFromColumnMetaData(History history, Integer columnNum) {
		String desc = "";
		if( history.getMetaData() != null  &&  history.getMetaData().getColumns() != null  && history.getMetaData().getColumns().size() > columnNum )
			desc = history.getMetaData().getColumns().get(columnNum).getDescription();
		return desc;
	}

	private KeyValType getTypeFromColumnMetaData(ColumnMetaData.Type columnType) {
		if( columnType.equals(ColumnMetaData.Type.Boolean) )
			return KeyValType.Flag;
		else if( columnType.equals(ColumnMetaData.Type.Float) )
			return KeyValType.Float;
		else if( columnType.equals(ColumnMetaData.Type.Integer) )
			return KeyValType.Integer;
		else
			return KeyValType.String;
	}
	
	private KeyValType getTypeFromColumnMetaData(History history, Integer columnNum) {
		ColumnMetaData.Type columnType = ColumnMetaData.Type.String;
		if( history.getMetaData() != null  &&  history.getMetaData().getColumns() != null  && history.getMetaData().getColumns().size() > columnNum )
			columnType = history.getMetaData().getColumns().get(columnNum).getType();
		
		return getTypeFromColumnMetaData(columnType);
	}
	
	protected String replaceBadCharsAndMetaDelimsInValue(String key, String value){
        value = removeQuotes(value);
        String delim = getDelimiterInMetadataHeader(key);
        
        StringBuilder s = new StringBuilder();
        for(int i=0; i < value.length(); i++) {
        	char c = value.charAt(i);
        	// WARNING:  Delimiter could be more than one character!
        	if( delim != null  &&  value.indexOf(delim,i) == i ) {
        		s.append(",");
        		i += (delim.length() - 1);
        	} else if( isASpecialChar(c) ) {
        		if( CharacterEncodingMethod.SIMPLE.equals(this.characterEncodingMethod) )
        			s.append(replaceCharSimple(c));
        		else if( CharacterEncodingMethod.URL_BASIC.equals(this.characterEncodingMethod) )
        			s.append(replaceCharUrlBasic(c));
        		else // CharacterEncodingMethod.URL_EXTENDED
        			s.append(replaceCharUrlExtended(c));
        	} else {
        		s.append(c);
        	}
        }
        return s.toString();
    }



	private boolean isASpecialChar(char c) {
		return c == ','
			|| c == '='
			|| c == ';'
			|| c == ' '
			|| c == ':'
			|| c == '%'
			|| c == '\r'
			|| c == '\n'
			|| c == '\t';
	}

	/** Replace comma or semicolon with pipe;  space with underscore;  equals with colon */
	private String replaceCharSimple(char c) {
		if(      c == ',' )  return "|";
		else if( c == ';' )  return "|";
		else if( c == ' ' )  return "_";
		else if( c == '=' )  return ":";
		else 				 return c + "";
	}

	private String replaceCharUrlBasic(char c) {
		if(      c == ',' )  return "%2C";
		else if( c == '=' )  return "%3D";
    	else if( c == ';' )  return "%3B";
    	else 				 return c + "";
 		
	}

	private String replaceCharUrlExtended(char c) {
		if(      c == ',' )  return "%2C";
		else if( c == '=' )  return "%3D";
    	else if( c == ';' )  return "%3B";
    	else if( c == ':' )  return "%3A";
    	else if( c == '%' )  return "%25";
    	else if( c == '\r')  return "%0D";
    	else if( c == '\n')  return "%0A";
    	else if( c == '\t')  return "%09";
    	else 				 return c + "";
	}

//	protected String replaceBadCharsAndMetaDelimsInValueOldWay(String key, String value){
//        value = removeQuotes(value);
//        String val2 = replaceMetaDelimiterWithComma(key, value);
//        // NOTE: We ACTUALLY DO want the object equality operator here to test whether the string
//        //       has changed due to the delimiter substitution, even it if went from 
//        //       "A,B,C" with delimiter "," then back to "A,B,C" again 
//        //       (string LOOKS identical, but it had actually replaced the delimiter, and thus the object is different)
//        boolean hasDelimiterBeenReplaced =  value != val2;
//        return replaceBadCharsInValueOldWay(val2, hasDelimiterBeenReplaced);
//    }
    
    private String removeQuotes(String value) {
    	if( value != null && value.startsWith("\"") && value.endsWith("\"") )
    		return value.substring(1, value.length()-1);
    	return value;
	}

	/*
    * ------------------------------------
    * VCF Spec for info:
    *
    * INFO additional information: (String, no white-space, semi-colons, or equals-signs permitted;
    * commas are permitted only as delimiters for lists of values)
    * INFO fields are encoded as a semicolon-separated series of short keys with optional values in the format:
    * <key>=<data>[,data].
    * Arbitrary keys are permitted, although the following sub-fields are reserved (albeit optional):
    AA : ancestral allele
    AC : allele count in genotypes, for each ALT allele, in the same order as listed
    AF : allele frequency for each ALT allele in the same order as listed: use this when estimated from primary data, not called genotypes
    AN : total number of alleles in called genotypes
    BQ : RMS base quality at this position
    CIGAR : cigar string describing how to align an alternate allele to the reference allele
    DB : dbSNP membership
    DP : combined depth across samples, e.g. DP=154
    END : end position of the variant described in this record (for use with symbolic alleles)
    H2 : membership in hapmap2
    H3 : membership in hapmap3
    MQ : RMS mapping quality, e.g. MQ=52
    MQ0 : Number of MAPQ == 0 reads covering this record
    NS : Number of samples with data
    SB : strand bias at this position
    SOMATIC : indicates that the record is a somatic mutation, for cancer genomics
    VALIDATED : validated by follow-up experiment
    1000G : membership in 1000 Genomes
    * ------------------------------------
    * Ex:  AN=3.0;bior.gene.name=MTHFR;isInDbsnp;alts=A,C,G,T
    * NOTE: If the delimiter has already been replaced with "," then we don't want to change it back!
    *       For example, if "|" was replaced with ",", then we don't "," turned back into "|" again
    */
//	private String replaceBadCharsInValueOldWay(String val, boolean hasDelimiterAlreadyBeenReplaced) {
//		final String LIST_DELIMITER = "|";
//		val = (hasDelimiterAlreadyBeenReplaced  ?  val  :  val.replaceAll(",",  "|"));
//		val = val.replace(" ",  "_")
//   			     .replace("=",  ":")
//   			     .replace(";",  LIST_DELIMITER);
//		return val;
//	}

/*
    // Look in the header metadata to see if a delimiter has been used for the values, and if so, replace that delimiter with commas
	private String replaceMetaDelimiterWithComma(String key, String val) {
		String delim = getDelimiterInMetadataHeader(key);
		if( delim == null )
			return val;
		return val.replace(delim, ",");
	}
	*/
	
	/** Get the delimiter from the metadata header if it exists (ex:  <##BIOR=....,Delimiter="|">).  If not found, return null */
	private String getDelimiterInMetadataHeader(String key) {
		if( key == null  ||  key.length() == 0 
		 || mBiorHeaderMap == null
		 || mBiorHeaderMap.get(key) == null
		 || mBiorHeaderMap.get(key).get(BiorMetaControlledVocabulary.DELIMITER.toString()) == null )
		{
			return null;
		} else {
			return mBiorHeaderMap.get(key).get(BiorMetaControlledVocabulary.DELIMITER.toString());
		}
	}



}
