package edu.mayo.bior.pipeline.VEP;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

/**
 * @author Michael Meiners (m054457)
 * Date created: Apr 25, 2013
 */
public class VepFunctions {
	
	private VepCsqComparator mVepCsqComparator = new VepCsqComparator();
	private List<String> mVepCsqHeaders = new ArrayList<String>();
	
	public static final String VEP_ERROR_MSG = "VEPERRORMessage";
	
	class Effect {
		public String term;
		public Double score;
	}

	/** Constructor that assumes it knows the default VEP CSQ headers
	 *  (use the other constructor when retrieving the CSQ headers from the returned ##INFO line from VEP) 
	 */
	public VepFunctions() {
		this(null);
	}
	
	/** Pass the list of VEP CSQ headers that are pipe-separated when returned in the ##INFO line from VEP.
	 *  Pass null or empty-list to use the defaults 
	 * @param vepCsqHeaders  List of CSQ headers that appears in the VEP ##INFO header */
	public VepFunctions(List<String> vepCsqHeaders) {
		mVepCsqHeaders = vepCsqHeaders;
		
		if( mVepCsqHeaders == null  ||  mVepCsqHeaders.size() == 0 ) {
			mVepCsqHeaders = new ArrayList<String>( Arrays.asList(
				"Allele",
				"Gene",
				"Feature",
				"Feature_type",
				"Consequence",
				"cDNA_position",
				"CDS_position",
				"Protein_position",
				"Amino_acids",
				"Codons",
				"Existing_variation",
				"HGNC",
				"DISTANCE",
				"SIFT",
				"PolyPhen",
				"CELL_TYPE"
				) );
		}
		
	}
	
	/** Turn CSQ result ("CSQ=x|y|z") into json ("{CSQ:{"a":"x","b":"y","c":"z"}}")  */
	public JsonArray vepCsqToJsonList(String vepCsq) {		
		JsonArray jsonArray = new JsonArray();		
		if (vepCsq.contains("VEPERR")) {
			JsonObject jObj = new JsonObject();
			jObj.addProperty("VEPMessage", VEP_ERROR_MSG);
			jObj.addProperty("Status", "VEP failed to assign function to this variant");
			jsonArray.add(jObj);
		} else {
			String[] vepCsqItems = vepCsq.replace("CSQ=", "").split(",");
			for(String csqItem : vepCsqItems) {
				jsonArray.add(vepCsqToJson(csqItem));
			}
		}
		
		return jsonArray;
			
	}
	
	
	
	/** Turn CSQ result into JSON
	 *  Ex CSQ: "A|ENSG00000154719|ENST00000307301|Transcript|missense_variant|1043|1001|334|T/M|aCg/aTg||MRPL39||tolerated(0.05)|benign(0.001)|"
	 *  Ex JSON:"{"Allele":"A","Gene":"ENSG00000154719","Feature":"ENST00000307301","Feature_type":"Transcript","Consequence":"missense_variant","cDNA_position":"1043","CDS_position":"1001","Protein_position":"334","Amino_acids":"T/M","Codons":"aCg/aTg","HGNC":"MRPL39","SIFT":"tolerated(0.05)","PolyPhen":"benign(0.001)","SIFT_TERM":"tolerated","SIFT_Score":0.05,"PolyPhen_TERM":"benign","PolyPhen_Score":0.001}"  */
	private JsonObject vepCsqToJson(String vepCsqItem) {
		String[] vepParts = vepCsqItem.split("\\|");
		JsonObject jsonObj = new JsonObject();

		// NOTE: the split ignores any empty columns at the end, so there may be fewer than 16 after split 
		// For ex: "a||b|||" will only have 3 columns
		for(int i=0; i < Math.min(vepParts.length, mVepCsqHeaders.size()); i++) {
			if( ! "".equals(vepParts[i]))
				jsonObj.addProperty(mVepCsqHeaders.get(i), vepParts[i]);
		}
		
		// Is SIFT present?
		boolean isSiftScoresGiven = vepParts.length > 13 && vepParts[13].contains("(") && vepParts[13].contains(")");
		if( isSiftScoresGiven ) {
			Effect sift = parseEffect(vepParts[13]);
			jsonObj.addProperty("SIFT_TERM", sift.term);
			jsonObj.addProperty("SIFT_Score", sift.score);
		}
		
		// Is PolyPhen present?
		boolean isPolyphenScoresGiven = vepParts.length > 14 && vepParts[14].contains("(") && vepParts[14].contains(")");
		if( isPolyphenScoresGiven ) {
			Effect polyphen = parseEffect(vepParts[14]);
			jsonObj.addProperty("PolyPhen_TERM", polyphen.term);
			jsonObj.addProperty("PolyPhen_Score", polyphen.score);
		}
		return jsonObj;
	}

	/** Convert string (ex: "tolerated(0.05)") to an Effect object */
	private Effect parseEffect(String effStr) {
		int idxOpenParen = effStr.indexOf("(");
		int idxCloseParen= effStr.indexOf(")");
		
		if(idxOpenParen == -1 || idxCloseParen == -1 )
			return null;
		
		Effect effect = new Effect();
		effect.term = effStr.substring(0, idxOpenParen);
		effect.score = Double.parseDouble(effStr.substring(idxOpenParen+1, idxCloseParen));
		return effect;
	}
	
	/** Given a list of vepCsqOutputs as JSON strings, return the one that has the worst outcome */
	public JsonObject getWorstCase(JsonArray jsonArray) {
		if( jsonArray == null  ||  jsonArray.size() == 0 )
			return null;
		
		List<JsonObject> jsonObjList = jsonArrayToList(jsonArray);
		Collections.sort(jsonObjList, mVepCsqComparator);
		
		// Even if there were no objects in the list, it will still have one object: a blank JSON object "{}"
		return jsonObjList.get(0);
	}
	
	protected List<JsonObject> jsonArrayToList(JsonArray jsonArray) {
		List<JsonObject> list = new ArrayList<JsonObject>();
		for(int i=0; i < jsonArray.size(); i++) {
			JsonObject jsonObj = jsonArray.get(i).getAsJsonObject();
			list.add(jsonObj);
		}
		return list;
	}
	

	
}
