package edu.mayo.bior.catalog.markdown.transformer.comparison;

import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;

import java.io.PrintWriter;
import java.io.StringWriter;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.chompTrailingNewlines;
import static edu.mayo.bior.catalog.markdown.transformer.comparison.ComparisonStatsImpl.NOT_AVAILABLE_MESG;
import static java.lang.String.format;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Builds a markdown table about catalog rows that were updated between previous/current.
 */
public class CatalogRowUpdateImpl implements MarkdownTransformer {

    static final String TEMPLATE_KEYWORD = "COMPARISON_STATISTICS_ROW_UPDATE_TABLE";

    private CatalogDiffStatistics stats;

    CatalogRowUpdateImpl(CatalogDiffStatistics stats) {
        this.stats = stats;
    }

    @Override
    public String transform(String markdown) throws TransformException {
        if (stats == null) {
            return markdown.replace(TEMPLATE_KEYWORD, NOT_AVAILABLE_MESG);
        } else {

            return markdown.replace(TEMPLATE_KEYWORD, chompTrailingNewlines(buildTable()));
        }
    }

    private String buildTable() {
        long intersectionRows =
                stats.getSummaryStats().getCatalogOldEntries() - stats.getSummaryStats().getCatalogEntriesRemoved();
        long changedRows =
                stats.getSummaryStats().getCatalogEntriesChanged()
                        - stats.getSummaryStats().getCatalogEntriesRemoved()
                        - stats.getSummaryStats().getCatalogEntriesAdded();
        long unchangedRows = intersectionRows - changedRows;

        StringWriter sWtr = new StringWriter();
        PrintWriter pWtr = new PrintWriter(sWtr);

        pWtr.println("Metric | Description | Count | % of Intersection");
        pWtr.println("--- | --- | --- | ---");

        pWtr.println(
                format("%s | %s | %d | %.1f%%",
                        "Intersection Rows",
                        "# of rows present in both previous and current catalogs",
                        intersectionRows,
                        (float) intersectionRows / (float) intersectionRows * 100
                )
        );

        pWtr.println(
                format("%s | %s | %d | %.1f%%",
                        "Unchanged Rows",
                        "intersection rows with no column changes",
                        unchangedRows,
                        (float) unchangedRows / (float) intersectionRows * 100
                )
        );

        pWtr.println(
                format("%s | %s | %d | %.1f%%",
                        "Updated Rows",
                        "intersection rows that had at least 1 column change",
                        changedRows,
                        (float) changedRows / (float) intersectionRows * 100
                )
        );
        closeQuietly(pWtr);
        closeQuietly(sWtr);
        return sWtr.toString();
    }
}