package edu.mayo.bior.catalog;

import java.io.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Map.Entry;

import edu.mayo.bior.catalog.verification.MessageLogger;
import edu.mayo.bior.catalog.verification.VerifyErrorCodes;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import edu.mayo.bior.catalog.verification.CatalogVerifier.VAL_TYPE;

public class CatalogDataSource
{
   private static final String SHORT_UNIQUE_NAME 	= DataSourceKey.ShortUniqueName.name();
   private static final String DESCRIPTION 			= DataSourceKey.Description.name();
   private static final String SOURCE 				= DataSourceKey.Source.name();
   private static final String DATASET				= DataSourceKey.Dataset.name();
   private static final String VERSION 				= DataSourceKey.Version.name();
   private static final String BUILD 				= DataSourceKey.Build.name();
   private static final String DATA_SOURCE_RELEASE_DATE = DataSourceKey.DataSourceReleaseDate.name();
   private static final String FORMAT 				= DataSourceKey.Format.name();

   private static final List<String> REQ_PROP_NAMES_1_1_0 = new ArrayList<String>();
   private static final List<String> REQ_PROP_NAMES_1_1_1 = new ArrayList<String>();
   private static final List<String> REQ_PROP_NAMES_1_1_2 = new ArrayList<String>();

   static
   {
	   REQ_PROP_NAMES_1_1_0.addAll(Arrays.asList(
			   SHORT_UNIQUE_NAME,
			   DESCRIPTION,
			   SOURCE,
			   VERSION,
			   BUILD,
			   FORMAT
	   ));
	   
	  REQ_PROP_NAMES_1_1_1.addAll(REQ_PROP_NAMES_1_1_0);
      REQ_PROP_NAMES_1_1_1.add(DATASET);

      REQ_PROP_NAMES_1_1_2.addAll(REQ_PROP_NAMES_1_1_1);
      REQ_PROP_NAMES_1_1_2.add(DATA_SOURCE_RELEASE_DATE);
   }


   private static final String FORMAT_1_1_0 = "1.1.0";
   private static final String FORMAT_1_1_1 = "1.1.1";
   private static final String FORMAT_1_1_2 = "1.1.2";
   private static final List<String> SUPPORTED_FORMATS = Arrays.asList(FORMAT_1_1_0, FORMAT_1_1_1, FORMAT_1_1_2);

   private Logger sLogger = Logger.getLogger(CatalogDataSource.class);

   private HumanBuildAssembly mHumanBuildValue = null;
   private File mDatasourceFile;

   private Map<String, String> mProps = new LinkedHashMap<String, String>();

   private static final String RELEASE_DATE_FORMAT = "yyyy-MM-dd";
   private static final SimpleDateFormat releaseDateFormatParser = new SimpleDateFormat(RELEASE_DATE_FORMAT);
   static
   {
      releaseDateFormatParser.setLenient(false);
   }

   public CatalogDataSource(File datasourceFile) throws CatalogFormatException
   {
      mDatasourceFile = datasourceFile;

      readProperties(datasourceFile);

      checkFormat(datasourceFile);

      checkRequiredProperties(datasourceFile);
   }

   private void readProperties(File datasourceFile) throws CatalogFormatException
   {
      if (datasourceFile == null)
      {
         throw new CatalogFormatException("Supplied null file to CatalogDataSource");
      }
      if (!datasourceFile.exists())
      {
         throw new CatalogFormatException(String.format("'%s' does not exist", datasourceFile.getPath()));
      }
      if (!datasourceFile.isFile())
      {
         throw new CatalogFormatException(String.format("'%s' is not a file", datasourceFile.getPath()));
      }
      if (!datasourceFile.canRead())
      {
         throw new CatalogFormatException(String.format("'%s' is not readable", datasourceFile.getPath()));
      }

      BufferedReader rdr = null;
      try
      {
         rdr = new BufferedReader(new FileReader(datasourceFile.getPath()));
         Properties properties = new Properties();
         properties.load(rdr);
         List<String> keysSeenMultipleTimes = new ArrayList<String>();
         for (String key : properties.stringPropertyNames())
         {
            String value = properties.getProperty(key);
            
            // Check if the value contains a double-quote which could mess up any processing of VCFs later.
            // If it does, put out a warning and convert to single-quote
            if( value.contains("\"") ) {
            	sLogger.error("Error: value for key [" + key + "] within the datasource.properties file must not contain a double-quote.  These will be replaced with single-quotes.  File: " + datasourceFile.getPath());
            	value = value.replaceAll("\"", "'");
            }
            
            for (String requiredPropName : REQ_PROP_NAMES_1_1_2)
            {
               if (requiredPropName.equalsIgnoreCase(key))
               {
                  if (mProps.containsKey(requiredPropName))
                  {
                     keysSeenMultipleTimes.add(requiredPropName);
                  }
                  else
                  {
                     mProps.put(requiredPropName, value);
                  }
               }
            }

            if (key.equals(CatalogDataSource.BUILD) && value != null)
            {
               mHumanBuildValue = HumanBuildAssembly.assemblyFromString(value);
            }
         }
         if (!keysSeenMultipleTimes.isEmpty())
         {
            String keyMsg = "had multiple instances of key";
            if (keysSeenMultipleTimes.size() != 1)
            {
               keyMsg += "s";
            }
            Collections.sort(keysSeenMultipleTimes);
            keyMsg += String.format(" '%s'", StringUtils.join(keysSeenMultipleTimes, ","));
            throw new CatalogFormatException(String.format("'%s' %s", datasourceFile.getPath(), keyMsg));
         }
      }

      catch (IOException e)
      {
         String msg = String.format("Problem reading data source properties file '%s'", datasourceFile.getPath());
         sLogger.error(msg);
         sLogger.error("Caused by: " + e.getMessage());
         throw new CatalogFormatException(msg, e);
      }
      finally
      {
         IOUtils.closeQuietly(rdr);
      }
   }

   private void checkFormat(File datasourceFile) throws CatalogFormatException
   {
      String format = getFormat();
      if (format == null)
      {
         String msg = String.format("Must supply %s property in '%s'", FORMAT, datasourceFile.getPath());
         throw new CatalogFormatException(msg);
      }
      if (!SUPPORTED_FORMATS.contains(format))
      {
         String msg = String.format("%s value '%s' from '%s' not in supported set '%s'",
            FORMAT, format,
            datasourceFile.getPath(), StringUtils.join(SUPPORTED_FORMATS, ", "));
         throw new CatalogFormatException(msg);
      }
   }

   private void checkRequiredProperties(File datasourceFile) throws CatalogFormatException
   {
      List<String> requiredProperties;
      if (is110Format())
      {
         requiredProperties = REQ_PROP_NAMES_1_1_0;
      }
      else if (is111Format())
      {
         requiredProperties = REQ_PROP_NAMES_1_1_1;
      }
      else if (is112Format())
      {
         requiredProperties = REQ_PROP_NAMES_1_1_2;
      }
      else
      {
         throw new RuntimeException("Programming error. Should have legal Format when checkRequiredProps is called.");
      }

      List<String> propertiesNotSupplied = new ArrayList<String>();
      for (String key : requiredProperties)
      {
         if (null == mProps.get(key))
         {
            propertiesNotSupplied.add(key);
         }
      }

      if (!propertiesNotSupplied.isEmpty())
      {
         Collections.sort(propertiesNotSupplied);
         String propertyDescription = "property";
         if (propertiesNotSupplied.size() > 1)
         {
            propertyDescription = "properties";
         }
         String msg = String.format("Required datasource %s '%s' not supplied in '%s'",
            propertyDescription, StringUtils.join(propertiesNotSupplied, ","), datasourceFile.getPath());
         throw new CatalogFormatException(msg);
      }
   }

   public File getFile() { return mDatasourceFile; }

   public String getShortUniqueName()
   {
      return getPropValue(SHORT_UNIQUE_NAME);
   }

   public String getDescription()
   {
      return getPropValue(DESCRIPTION);
   }

   public String getSource()
   {
      return getPropValue(SOURCE);
   }

   public String getDataset()
   {
      return getPropValue(DATASET);
   }

   public String getVersion()
   {
      return getPropValue(VERSION);
   }

   public String getBuild()
   {
      return getPropValue(BUILD);
   }

   public String getDataSourceReleaseDate()
   {
      return getPropValue(DATA_SOURCE_RELEASE_DATE);
   }

   /**
    * Parses string from {@link CatalogDataSource#getDataSourceReleaseDate()} into a {@link Date}
    * @return A {@link Date} representation of the dataSourceReleaseDate
    * @throws CatalogFormatException thrown if the dataSourceReleaseDate has an invalid format
    */
   public Date getDataSourceReleaseDateAsDateObject() throws CatalogFormatException {
      try {
         return releaseDateFormatParser.parse(getDataSourceReleaseDate());
      } catch (ParseException pe) {
         String mesg = String.format("%s value '%s' is incorrectly formatted (%s) in file %s",
                 DataSourceKey.DataSourceReleaseDate.name(),
                 getDataSourceReleaseDate(),
                 RELEASE_DATE_FORMAT,
                 getFile().getAbsolutePath());
         throw new CatalogFormatException(mesg, pe);
      }
   }

   public String getFormat()
   {
      return getPropValue(FORMAT);
   }

   private String getPropValue(String key)
   {
      String value = mProps.get(key);
      if (value == null)
      {
         return null;
      }
      return value.trim();
   }

   // We are guaranteed that getFormat() won't return null because it's checked in constructor
   public boolean is110Format()
   {
      return FORMAT_1_1_0.equals(getFormat());
   }

   // We are guaranteed that getFormat() won't return null because it's checked in constructor
   public boolean is111Format()
   {
      return FORMAT_1_1_1.equals(getFormat());
   }

   // We are guaranteed that getFormat() won't return null because it's checked in constructor
   public boolean is112Format()
   {
      return FORMAT_1_1_2.equals(getFormat());
   }

   public List<String> getPropertiesWithBlankValue()
   {
      List<String> propertiesWithBlankValue = new ArrayList<String>();
      for (Entry<String, String> entrySet : mProps.entrySet())
      {
         if (StringUtils.isBlank(entrySet.getValue()))
         {
            propertiesWithBlankValue.add(entrySet.getKey());
         }
      }
      Collections.sort(propertiesWithBlankValue);
      return propertiesWithBlankValue;
   }

   public void verify(VAL_TYPE valType, MessageLogger logger)
   {
      List<String> propsWithBlankValue = getPropertiesWithBlankValue();
      List<String> keysToComplainAbout = new ArrayList<String>();
      if (propsWithBlankValue.contains(VERSION))
      {
         keysToComplainAbout.add(VERSION);
      }
      if (valType == VAL_TYPE.STRICT)
      {
         if (propsWithBlankValue.contains(BUILD))
         {
            keysToComplainAbout.add(BUILD);
         }
         if (propsWithBlankValue.contains(DESCRIPTION))
         {
            keysToComplainAbout.add(DESCRIPTION);
         }
      }
      if (!keysToComplainAbout.isEmpty())
      {
         String description = "Property";
         if (keysToComplainAbout.size() > 1)
         {
            description = "Properties";
         }
         String msg = String.format("%s '%s' in '%s' should have a value",
                                    description, StringUtils.join(keysToComplainAbout, ","), mDatasourceFile.getPath());
         logger.logWarning(msg);
      }

      if (getHumanBuildAssembly() == null)
      {
         String msg = String.format("Human Genome Assembly was not determined from property '%s' in '%s'",
                                    BUILD, mDatasourceFile.getPath());
         logger.logWarning(msg);
      }
      else
      {
         logger.logInfo("Catalog is based on Human Genome Build '" + getHumanBuildAssembly().name() +
               "'. Will check chromosomes, positions, and reference alleles accordingly.");
      }

      String releaseDate = getDataSourceReleaseDate();
      try
      {
         if (releaseDate != null && !("".equals(releaseDate)))
         {
            releaseDateFormatParser.parse(getDataSourceReleaseDate());
         }
      }
      catch (ParseException e)
      {
         String msg = String.format("%s value '%s' is either incorrectly formatted (%s) or a date that does not " +
            "exist. Details: %s",
            DataSourceKey.DataSourceReleaseDate, releaseDate, RELEASE_DATE_FORMAT, e.getMessage());
         logger.logError(msg, VerifyErrorCodes.DATASOURCE_PROPERTIES_BAD_RELEASE_DATE);
      }

   }

   /**
    * getHumanBuildAssembly: If this catalog is not a human catalog, null will be returned. Otherwise,
    * the value found in the data source properties file for the key DATASET
    * will be compared to the well-known Human Genome Build values.
    *
    * @return If this catalog is not a human catalog, null will be returned.
    */
   public HumanBuildAssembly getHumanBuildAssembly()
   {
      return mHumanBuildValue;
   }

   public boolean isHumanCatalog()
   {
      return mHumanBuildValue != null;
   }
}
