package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import edu.mayo.pipes.history.ColumnMetaDataVerificationMessages;
import org.apache.commons.io.FileUtils;

import edu.mayo.bior.pipeline.exception.CatalogMetadataInputException;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaDataOperations;
import edu.mayo.pipes.history.ColumnMetaData.Headers;
import edu.mayo.pipes.history.ColumnMetaData.Type;

/** 3-way merge of columns.tsv files between the current (possibly manually edited) file, 
 *  the default file from crawling the newly built catalog,
 *  and the previous catalog's file
 *  @author Michael Meiners (m05445) - 2016-05-04 */
public class MergeColumnsTsv {
	
	private File       mCatalogDir;
	private String     mCatalogShortName;
	private File       mPreviousCatalogColumnsTsv;
	private StepLogger mStepLogger;
	private boolean    mIsUpdateNecessary = false;

	public MergeColumnsTsv(File catalogDir, String catalogShortName, File previousCatalogColumnsTsv, StepLogger stepLogger) {
		mCatalogDir = catalogDir;
		mCatalogShortName = catalogShortName;
		mPreviousCatalogColumnsTsv = previousCatalogColumnsTsv;
		mStepLogger = stepLogger;
	}
	
	
	public void mergeColumnsTsv() throws IOException, CatalogMetadataInputException {
		File currentColumnsFile  	= new File(mCatalogDir, mCatalogShortName + ".columns.tsv");
		File defaultColumnsFile  	= new File(mCatalogDir.getCanonicalPath() + "/" + BuildCatalog.BUILD_SUBDIR, mCatalogShortName + ".columns.tsv.default");
		
		ColumnMetaDataOperations colMetaOps = new ColumnMetaDataOperations(currentColumnsFile);
		HashMap<String,ColumnMetaData> currentColInfo = colMetaOps.load();
		HashMap<String,ColumnMetaData> defaultColInfo = new ColumnMetaDataOperations(defaultColumnsFile).load();
		HashMap<String,ColumnMetaData> prevCtgColInfo = new HashMap<String,ColumnMetaData>();
		if( mPreviousCatalogColumnsTsv != null  &&  mPreviousCatalogColumnsTsv.exists() )
			prevCtgColInfo = new ColumnMetaDataOperations(mPreviousCatalogColumnsTsv).load();

		List<String> defaultExtras = new ArrayList<String>(defaultColInfo.keySet());
		List<String> prevExtras    = new ArrayList<String>(prevCtgColInfo.keySet());
		List<String> currColNames    = new ArrayList<String>(currentColInfo.keySet());
		
		BackupUtils.backupCurrentFile(currentColumnsFile);
		
		// Loop thru each key in the currentColumns file,
		//   Then loop thru each columnHeader in the comparing it against the defaults and the previous catalog entries
		// Five columns:  ColumnName,  Type,  Count,  Description,  HumanReadableName
		List<ColumnMetaData> colMetaList = new ArrayList<ColumnMetaData>();
		for(String colKey : currColNames) {
			colMetaList.add(mergeColumn(colKey, currentColInfo, defaultColInfo, prevCtgColInfo));
			
			// If the key is not in the defaults, then warn the user
			if( defaultColInfo.get(colKey) == null ) {
				mStepLogger.log("Warning: (columns.tsv): Extra key?  '" + colKey  + "' is in the current columns.tsv, but was not detected in the catalog (not in columns.tsv.defaults).");
				mIsUpdateNecessary = true;
			}
			
			// If the key is not in the previous catalog, then warn the user
			if( defaultColInfo.get(colKey) == null  &&  mPreviousCatalogColumnsTsv != null ) {
				mStepLogger.log("Warning: (columns.tsv): Extra key?  '" + colKey  + "' is in the current columns.tsv, but not in the previous catalog's columns.tsv.");
				mIsUpdateNecessary = true;
			}
			
			// Remove the columns processed here from the list of defaults and previous catalog columns
			defaultExtras.remove(colKey);
			prevExtras.remove(colKey);
		}
		
		//testPrint("Current:", colMetaList);
		//testPrint("Default:", new ArrayList<ColumnMetaData>(defaultColInfo.values()));
		//testPrint("Previous:", new ArrayList<ColumnMetaData>(prevCtgColInfo.values()));
		

		// Any extra "default" columns should be added to the current list as these were detected when crawling the catalog
		// NOTE: We don't want to remove any columns as the user may be working on a subset of a full catalog and therefore may not have the full list yet.
		colMetaList.addAll(getDefaultsNotInCurrent(defaultExtras, defaultColInfo));
		logExtraDefaultColumns(defaultExtras, defaultColInfo);

		logAnyExtraPreviousColumns(prevExtras, prevCtgColInfo);
		
		save(currentColumnsFile, colMetaList, colMetaOps);
	}


	private void testPrint(String msg, List<ColumnMetaData> colMetaList) {
		System.out.println("================================");
		System.out.println(msg);
		for(ColumnMetaData col : colMetaList) {
			System.out.println(col.toString());
		}
	}


	private List<ColumnMetaData> getDefaultsNotInCurrent(
			List<String> defaultColNames,
			HashMap<String, ColumnMetaData> defaultColInfo)
	{
		List<ColumnMetaData> defaultsNotInCurrent = new ArrayList<ColumnMetaData>();
		for(String defaultCol : defaultColNames) {
			defaultsNotInCurrent.add(defaultColInfo.get(defaultCol));
		}
		return defaultsNotInCurrent;
	}


	/** Log any columns that were not covered from previous catalog 
	 * @param prevCtgColInfo */
	private void logAnyExtraPreviousColumns(List<String> prevCtgColNames, HashMap<String, ColumnMetaData> prevCtgColInfo) {
		if( prevCtgColNames.size() > 0 ) {
			mIsUpdateNecessary = true;
			String msg = "Warning: (columns.tsv): These columns were present in the previous catalog's columns.tsv but not in the current columns.tsv:";
			mStepLogger.log(msg);
			for(String colName : prevCtgColNames) {
				mStepLogger.log("    " + prevCtgColInfo.get(colName));
			}
		}
	}


	/** Log any columns that were in the defaults (from crawling the catalog), but were not in the "current" list 
	 * @param defaultColInfo */
	private  void  logExtraDefaultColumns(List<String> defaultColNames, HashMap<String, ColumnMetaData> defaultColInfo) {
		if( defaultColNames.size() > 0 ) {
			mIsUpdateNecessary = true;
			String msg = "Warning: (columns.tsv): Some columns were detected in the catalog data (default columns.tsv) but not in the current columns.tsv.  These will be added to the current list.";
			mStepLogger.log(msg);
			for(String colName : defaultColNames) {
				mStepLogger.log("    " + defaultColInfo.get(colName));
			}
		}
	}

	
	private void save(File currentColumnsFile, List<ColumnMetaData> colMetaList, ColumnMetaDataOperations colMetaOps) throws IOException {
		sortColumnsById(colMetaList);
		
		colMetaOps.save(colMetaList, /*isForceOverwrite=*/true);

		String path = currentColumnsFile.getPath();
		if( BackupUtils.isCurrentSameAsLastBackup(currentColumnsFile) ) {
			mStepLogger.logAndSummary(String.format("No changes made to '%s'", path));
			BackupUtils.removeLastBackup(currentColumnsFile);
		} else {
			mStepLogger.logAndSummary(String.format("Some changes made to '%s'", path));
		}

		ColumnMetaDataVerificationMessages msgs = colMetaOps.verifyContents();
		if (msgs.getErrors().size() > 0 || msgs.getWarnings().size() > 0) {
			mStepLogger.logAndSummary(String.format("Columns.tsv file '%s' has some issues. See verify output for details.", path));
		}
	}




	private void sortColumnsById(List<ColumnMetaData> colMetaList) {
		Collections.sort(colMetaList, new Comparator<ColumnMetaData>() {
			public int compare(ColumnMetaData o1, ColumnMetaData o2) {
				return o1.getColumnName().compareToIgnoreCase(o2.getColumnName());
			}
		});
	}


	private ColumnMetaData mergeColumn( String colName,
			HashMap<String,ColumnMetaData>  currentColInfo,
			HashMap<String,ColumnMetaData>  defaultColInfo,
			HashMap<String,ColumnMetaData>  prevCtgColInfo)
	{
		ColumnMetaData colMeta = new ColumnMetaData(colName);
		
		ColumnMetaData currentColMeta = currentColInfo.get(colName);
		ColumnMetaData defaultColMeta = defaultColInfo.get(colName);
		ColumnMetaData prevCtgColMeta = prevCtgColInfo.get(colName);
		
		// Use the current type if available, but warn if the default or previousCatalog types are different
		Type currentType = Type.valueOf(getValue(colName, Headers.Type, currentColInfo));
		colMeta.setType(currentType);
		checkCurrentVsOthers(colName, Headers.Type, currentColMeta, defaultColMeta, prevCtgColMeta);
		
		// Use the current count, but warn if the default or previousCatalog types were different
		String currentCount =  getValue(Headers.Count, currentColMeta);
		colMeta.setCount(currentCount);
		checkCurrentVsOthers(colName, Headers.Count, currentColMeta, defaultColMeta, prevCtgColMeta);

		colMeta.setDescription(getBestDescription(currentColMeta, defaultColMeta, prevCtgColMeta));
		
		colMeta.setHumanReadableName(getBestHumanReadableName(currentColMeta, defaultColMeta, prevCtgColMeta));

		return colMeta;
	}

	

	/** Get the best description for the current columns.tsv
	  * Use the current description if it is given, 
	  * else use the previous description if it is given
	  * else use the default description if it is given
	  * else return empty string ("").    */
	private String getBestDescription(ColumnMetaData currentColMeta, ColumnMetaData defaultColMeta, ColumnMetaData prevCtgColMeta) {
		String currentDescription = getValue(Headers.Description, currentColMeta);
		String defaultDescription = getValue(Headers.Description, defaultColMeta);
		String prevCtgDescription = getValue(Headers.Description, prevCtgColMeta);
		
		if( isGiven(currentDescription) ) {
			return currentDescription;
		}

		if( isGiven(prevCtgDescription) ) {
			mStepLogger.log("Warning: (columns.tsv): Using the previous catalog's description for column '" + currentColMeta.getColumnName()  + "'");
			mIsUpdateNecessary = true;
			return prevCtgDescription;
		}
		
		if( isGiven(defaultDescription) ) { 
			mStepLogger.log("Warning: (columns.tsv): Using the default catalog description for column '" + currentColMeta.getColumnName()  + "'");
			mIsUpdateNecessary = true;
			return defaultDescription;
		}
		
		return "";
	}
	
	/** Get the best HumanReadableName for the current columns.tsv
	  * Use the current one if it is given and NOT the same as the columnName 
	  * else use the previous description if it is given and NOT the same as the columnName
	  * else use the default description if it is given and NOT the same as the columnName
	  * else use the columnName (this is the default when crawling the catalog).    */
	private String getBestHumanReadableName(ColumnMetaData currentColMeta, ColumnMetaData defaultColMeta, ColumnMetaData prevCtgColMeta) {
		String currentHumanReadableName = getValue(Headers.HumanReadableName, currentColMeta);
		String defaultHumanReadableName = getValue(Headers.HumanReadableName, defaultColMeta);
		String prevCtgHumanReadableName = getValue(Headers.HumanReadableName, prevCtgColMeta);

		String colName = currentColMeta.getColumnName();
		
		if( isGiven(currentHumanReadableName) && ! colName.equals(currentHumanReadableName) ) {
			return currentHumanReadableName;
		}

		if( isGiven(prevCtgHumanReadableName) && ! colName.equals(prevCtgHumanReadableName) ) {
			mStepLogger.log("Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column '" + currentColMeta.getColumnName()  + "'");
			mIsUpdateNecessary = true;
			return prevCtgHumanReadableName;
		}
			
		if( isGiven(defaultHumanReadableName) && ! colName.equals(defaultHumanReadableName) ) {
			mStepLogger.log("Warning: (columns.tsv): Using the default catalog's HumanReadableName for column '" + currentColMeta.getColumnName()  + "'");
			mIsUpdateNecessary = true;
			return defaultHumanReadableName;
		}
		
		return colName;
	}


	/** Log a warning if the current value is different from the default or previous catalog value */
	private void checkCurrentVsOthers(
			String colName,
			ColumnMetaData.Headers colHeader,
			ColumnMetaData current,
			ColumnMetaData defaults,
			ColumnMetaData previous)
	{
		// Warn if the default or previousCatalog values were different
		String currentVal = getValue(colHeader, current);
		String defaultVal = getValue(colHeader, defaults);
		String prevCtgVal = getValue(colHeader, previous);
		
		if( defaultVal != null  &&  ! currentVal.equals(defaultVal) ) {
			mStepLogger.log("Warning: (columns.tsv): Column " + colName + ":  The current *" + colHeader + "* is different from the default that was calculated from crawling all data in the catalog.");
			mStepLogger.log("    current " + colHeader + ": " + (currentVal == null ? "(none)" : currentVal));
			mStepLogger.log("    default " + colHeader + ": " + (defaultVal == null ? "(none)" : defaultVal));
		}
		
		if( mPreviousCatalogColumnsTsv != null  &&  prevCtgVal != null  &&  ! currentVal.equals(prevCtgVal) ) {
			mStepLogger.log("Warning: (columns.tsv): Column " + colName + ": The current *" + colHeader + "* is different from the same column in the previous catalog.");
			mStepLogger.log("    current  " + colHeader + ": " + (currentVal == null ? "(none)" : currentVal));
			mStepLogger.log("    previous " + colHeader + ": " + (prevCtgVal == null ? "(none)" : prevCtgVal));
		}
	}
	


	private boolean isGiven(String currentVal) {
		return currentVal != null  &&  currentVal.trim().length() > 0  &&  ! currentVal.equals(".");
	}

	private String getValue(String columnName, ColumnMetaData.Headers colHeader,	HashMap<String,ColumnMetaData> colNameToMetaMap) {
		if( colNameToMetaMap == null ) {
			return null;
		}

		ColumnMetaData colMeta = colNameToMetaMap.get(columnName);
		if( colMeta == null ) {
			return null;
		}

		return getValue(colHeader, colMeta);
	}
	
	private String getValue(Headers colHeader, ColumnMetaData colMeta) {
		if( colMeta == null )  {
			return null;
		}
				
		if( colHeader.equals(ColumnMetaData.Headers.ColumnName) ) {
			return colMeta.getColumnName();
		} else if( colHeader.equals(ColumnMetaData.Headers.Type) ) {
			return colMeta.getType().toString();
		} else if( colHeader.equals(ColumnMetaData.Headers.Count) ) {
			return colMeta.getCount();
		} else if( colHeader.equals(ColumnMetaData.Headers.Description) ) {
			return colMeta.getDescription();
		} else { //colHeader.equals(ColHeader.HumanReadableName)
			return colMeta.getHumanReadableName();
		}
	}
}
